/*
 * Tony Givargis - 03/30/96
 *
 * PC serial communications interface.
 */

#ifndef __port_h__
#define __port_h__

#include <stdlib.h>
#include <dos.h>

/*
 * Port number.
 */
#define COM_1                   0x0000
#define COM_2                   0x0001
#define COM_3                   0x0002
#define COM_4                   0x0003

/*
 * Port initialization param.
 */
#define	CHAR_5			0x00
#define CHAR_6                  0x01
#define CHAR_7                  0x02
#define CHAR_8                  0x03
#define STOP_1			0x00
#define STOP_2                  0x04
#define PARITY_N		0x00
#define PARITY_O                0x20
#define PARITY_E                0x30
#define PARITY_M                0x28
#define PARITY_S                0x38
#define BREAK_ON		0x00
#define BREAK_OFF               0x40

/*
 * Quick port initialization param.
 */
#define PARAM_N81		(CHAR_8|STOP_1|PARITY_N)
#define PARAM_O81               (CHAR_8|STOP_1|PARITY_O)

/*
 * Baud rate.
 */
#define BAUD_300		0x0180
#define BAUD_1200      		0x0060
#define BAUD_2400    		0x0030
#define BAUD_9600      		0x000c
#define BAUD_19200     		0x0006
#define BAUD_38400     		0x0003
#define BAUD_57600		0x0002
#define BAUD_115200		0x0001

/*
 * Base com i/o.
 */
#define C1_BASE			0x03f8
#define C2_BASE			0x02f8
#define C3_BASE			0x03e8
#define C4_BASE                 0x02e8

/*
 * IRQ defines.
 */
#define C1_INT          	0x0c
#define C2_INT                  0x0b
#define C3_INT		        0x0c
#define C4_INT                  0x0b

/*
 * PIC's register and mask defines.
 */
#define C1_PICM			0xef
#define C2_PICM			0xf7
#define C3_PICM                 0xef
#define C4_PICM                 0xf7

/*
 * UART register defines.
 */
#define REG_THR			0
#define REG_RDR         	0
#define REG_BRL         	0
#define REG_IER         	1
#define REG_BRH         	1
#define REG_IIR     		2
#define REG_LCR     		3
#define REG_MCR       		4
#define REG_LSR       		5
#define REG_MSR         	6

/*
 * Bit set defines.
 */
#define BIT_SET_0       	0x01
#define BIT_SET_1       	0x02
#define BIT_SET_2       	0x04
#define BIT_SET_3       	0x08
#define BIT_SET_4       	0x10
#define BIT_SET_5       	0x20
#define BIT_SET_6       	0x40
#define BIT_SET_7       	0x80

/*
 * Bit clear defines.
 */
#define BIT_CLR_0       	0xfe
#define BIT_CLR_1       	0xfd
#define BIT_CLR_2       	0xfb
#define BIT_CLR_3       	0xf7
#define BIT_CLR_4       	0xef
#define BIT_CLR_5       	0xdf
#define BIT_CLR_6       	0xbf
#define BIT_CLR_7		0x7f

/*
 * The queue abstract data type declaration.
 */
typedef struct queue_struct
{
  unsigned size;
  unsigned count;
  unsigned head;
  unsigned tail;
  unsigned char* mem;
} queue_t;

/*
 * Pointer to interrupt handler function.
 */
typedef void interrupt far (*intr_t) (void);

/*
 * Port's interrupt service functions.
 */
void interrupt far c1_intr(void);
void interrupt far c2_intr(void);
void interrupt far c3_intr(void);
void interrupt far c4_intr(void);

/*
 * Opens port for I/O. Returns 0 on error.
 */
unsigned
init_port(unsigned port, unsigned param, unsigned baud, unsigned qsize);

/*
 * Closes port.
 */
void
close_port(unsigned port);

/*
 * Returns the number of bytes in port's receive queue.
 */
unsigned
qsize_port(unsigned port);

/*
 * Clears port's crecieve queue.
 */
void
flush_port(unsigned port);

/*
 * Reads n bytes from port into buff.
 */
void
read_port(unsigned port, void* buff, unsigned n);

/*
 * Writes n bytes from buff to port.
 */
void
write_port(unsigned port, void* buff, unsigned n);

#endif