/*
 * Tony Givargis - 03/30/96
 *
 * PC serial communications interface.
 */

#include "port.h"

/*
 * Port's base address lookup table.
 */
unsigned base_table[4] = { C1_BASE, C2_BASE, C3_BASE, C4_BASE };

/*
 * Port's interrupt vector number lookup table.
 */
unsigned int_table[4] = { C1_INT, C2_INT, C3_INT, C4_INT };

/*
 * Port's old interrupt vector lookup table.
 */
intr_t oldv_table[4] = { 0, 0, 0, 0 };

/*
 * Port's new interrupt vector lookup table.
 */
intr_t newv_table[4] = { c1_intr, c2_intr, c3_intr, c4_intr };

/*
 * PIC's mask lookup table.
 */
unsigned char picm_table[4] = { C1_PICM, C2_PICM, C3_PICM, C4_PICM };

/*
 * Port's queue lookup table.
 */
queue_t queue_table[4];

/*
 * Initializes the UART
 */
void
init_uart_chip(unsigned port, unsigned param, unsigned baud)
{
  unsigned char reg = 0;
  unsigned char picm = picm_table[port];
  unsigned base = base_table[port];

  /* Set port settings. */
  (void) outp(base_table[port]+REG_LCR, param);

  /* Set port baud rate. */
  reg = inp(base+REG_LCR);
  (void) outp(base+REG_LCR, reg | BIT_SET_7);
  (void) outp(base+REG_BRH, baud >> 8);
  (void) outp(base+REG_BRL, baud & 0x0f);
  reg = inp(base+REG_LCR);
  (void) outp(base+REG_LCR, reg & BIT_CLR_7);

  /* Disable all interrupts. */
  asm cli

  /* Set DTR, RTS, USR1, USR2 lines. */
  (void) outp(base+REG_MCR, 0x0f);

  /* Set PIC's interrupt enable register. */
  reg = inp(0x21);
  (void) outp(0x21, reg & picm);

  /* Set UART's interrupt enable register. */
  (void) outp(base+REG_IER, BIT_SET_0);

  /* Enable all interrupts. */
  asm sti
}

/*
 * Opens port for I/O. Returns 0 on error.
 */
unsigned
init_port(unsigned port, unsigned param, unsigned baud, unsigned qsize)
{
  /* Create the input queues. */
  queue_table[port].mem = (unsigned char*) malloc(qsize);
  queue_table[port].size = qsize;
  flush_port(port);

  /* Check allocation. */
  if( !queue_table[port].mem )
    return 0;

  /* Save the old handler and install the new handler. */
  oldv_table[port] = getvect(int_table[port]);
  setvect(int_table[port], newv_table[port]);

  /* Protocol settings. */
  init_uart_chip(port, param, baud);
  return 1;
}

/*
 * Closes port.
 */
void
close_port(unsigned port)
{
  unsigned char reg = 0;

  /* Disable all interrupts. */
  asm cli

  /* Clear DTR, RTS, USR1, USR2 lines. */
  (void) outp(base_table[port]+REG_MCR, 0x00);

  /* Clear PIC's interrupt enable register. */
  reg = inp(0x21);
  (void) outp(0x21, reg | ~picm_table[port]);

  /* Enable all interrupts. */
  asm sti

  /* Restore the old handler. */
  setvect(int_table[port], oldv_table[port]);

  /* Destroy the input queue. */
  free(queue_table[port].mem);
}

/*
 * Returns the number of bytes in port's receive queue.
 */
unsigned
qsize_port(unsigned port)
{
  return queue_table[port].count;
}

/*
 * Clears port's receive queue.
 */
void
flush_port(unsigned port)
{
  queue_table[port].count = 0;
  queue_table[port].head = 0;
  queue_table[port].tail = 0;
}

/*
 * Reads n bytes from port into buff.
 */
void
read_port(unsigned port, void* buff, unsigned n)
{
  queue_t* q = queue_table+port;
  unsigned char* b = (unsigned char*) buff;

  while( n-- )
  {
    /* Quickly read the queue. */
    *b++ = q->mem[q->head];
    q->head = (q->head == q->size) ? 0 : q->head+1;
    q->count--;
  }
}

/*
 * Writes n bytes from buff to port.
 */
void
write_port(unsigned port, void* buff, unsigned n)
{
  unsigned char reg = 0;
  unsigned char* b = (unsigned char*) buff;

  while( n-- )
  {
    do
    {
      reg = inp(base_table[port]+REG_LSR);
    }
    while( !(reg & BIT_SET_5) );
    (void) outp(base_table[port]+REG_THR, *b++);
  }
}

/*
 * Interrupt service function #1.
 */
void interrupt far
c1_intr(void)
{
  static unsigned char data;

  /* Stop the modem. */
  asm {
	mov  dx, C1_BASE+REG_MCR
	in   al, dx
	and  al, BIT_CLR_1
      }

  /* Read the characters. */
  asm {
	mov  dx, C1_BASE+REG_RDR
	in   al, dx
	mov  data, al
      }

  /* Quickly store it into the queue. */
  queue_table[0].mem[queue_table[0].tail] = data;
  queue_table[0].tail = (queue_table[0].tail == queue_table[0].size)
		      ? 0 : queue_table[0].tail+1;
  queue_table[0].count++;

  /* Send end of interrupt. */
  asm {
	mov  al, BIT_SET_5
	out  0x20, al
      }

  /* Start the modem. */
  asm {
	mov  dx, C1_BASE+REG_MCR
	in   al, dx
	or   al, BIT_SET_1
	out  dx, al
      }
}

/*
 * Interrupt service function #2.
 */
void interrupt far
c2_intr(void)
{
  static unsigned char data;

  /* Stop the modem. */
  asm {
	mov  dx, C2_BASE+REG_MCR
	in   al, dx
	and  al, BIT_CLR_1
      }

  /* Read the characters. */
  asm {
	mov  dx, C2_BASE+REG_RDR
	in   al, dx
	mov  data, al
      }

  /* Quickly store it into the queue. */
  queue_table[1].mem[queue_table[1].tail] = data;
  queue_table[1].tail = (queue_table[1].tail == queue_table[1].size)
		      ? 0 : queue_table[1].tail+1;
  queue_table[1].count++;

  /* Send end of interrupt. */
  asm {
	mov  al, BIT_SET_5
	out  0x20, al
      }

  /* Start the modem. */
  asm {
	mov  dx, C2_BASE+REG_MCR
	in   al, dx
	or   al, BIT_SET_1
	out  dx, al
      }
}

/*
 * Interrupt service function #3.
 */
void interrupt far
c3_intr(void)
{
  static unsigned char data;

  /* Stop the modem. */
  asm {
	mov  dx, C3_BASE+REG_MCR
	in   al, dx
	and  al, BIT_CLR_1
      }

  /* Read the characters. */
  asm {
	mov  dx, C3_BASE+REG_RDR
	in   al, dx
	mov  data, al
      }

  /* Quickly store it into the queue. */
  queue_table[2].mem[queue_table[2].tail] = data;
  queue_table[2].tail = (queue_table[2].tail == queue_table[2].size)
		      ? 0 : queue_table[2].tail+1;
  queue_table[2].count++;

  /* Send end of interrupt. */
  asm {
	mov  al, BIT_SET_5
	out  0x20, al
      }

  /* Start the modem. */
  asm {
	mov  dx, C3_BASE+REG_MCR
	in   al, dx
	or   al, BIT_SET_1
	out  dx, al
      }
}

/*
 * Interrupt service function #4.
 */
void interrupt far
c4_intr(void)
{
  static unsigned char data;

  /* Stop the modem. */
  asm {
	mov  dx, C4_BASE+REG_MCR
	in   al, dx
	and  al, BIT_CLR_1
      }

  /* Read the characters. */
  asm {
	mov  dx, C4_BASE+REG_RDR
	in   al, dx
	mov  data, al
      }

  /* Quickly store it into the queue. */
  queue_table[3].mem[queue_table[3].tail] = data;
  queue_table[3].tail = (queue_table[3].tail == queue_table[3].size)
		      ? 0 : queue_table[3].tail+1;
  queue_table[3].count++;

  /* Send end of interrupt. */
  asm {
	mov  al, BIT_SET_5
	out  0x20, al
      }

  /* Start the modem. */
  asm {
	mov  dx, C4_BASE+REG_MCR
	in   al, dx
	or   al, BIT_SET_1
	out  dx, al
      }
}