#include <stdio.h>
#include "PCSerial.h"

/******************************************************************************
PCSerial.c : PC Serial communication library using handshaking
When the device writes to the PC :
It waits for DTR to be low, sends the byte of data and then waits for RTS 
to be low. (the PC waits for a bit after asserting RTS so the device has
time to sample it.)
When the device reads from the PC :
It first sets DSR to 0, waits until it receives the byte of data and then
sets CTS to 0. The PC waits for both of them to be high before proceeding.
This avoids a race condition.
In both cases active low logic applies (from the device point of view), 
assuming a direct connection via an RS232-TTL driver/receiver chip such as
the LT1130CN (Linear Technology)
*******************************************************************************/

static int CL_PORT_ADS;            // holds the COM port address

static void pc_delay(int t)
{
   int i,j;
   for(i=0;i<t;++i)
      for(j=0;j<t;++j);
}

// Get a byte from the I/O port specified
static uchar _inp(uint addr)
{
  uchar rec_byte;
  asm {
	mov dx, addr
	in  al,dx
	mov rec_byte,al
      }
  return rec_byte;
}

// Send a byte to the I/O port specified
static void _outp(uint addr, uchar send_byte)
{
  asm {
	mov dx,addr
	mov al,send_byte
	out dx,al
      }
}

// Set the PC to the baud rate specified
static void baudset(long baudrate)
{
  uint divisor;
  uchar lsb, msb;

  divisor = 115200l / baudrate;

  msb = divisor >> 8;
  lsb = ( divisor << 8 )  >> 8;

  _outp(CL_REG_LCONT, CL_LCONT_DLAB);

  _outp(CL_PORT_ADS, lsb);

  _outp(CL_PORT_ADS+1, msb);
}

// Set the parity, data bits and stop bits specified
static void comparm(int parity, int stop, int databits)
{
  uchar parmbyte;

  parmbyte = databits - 5;

  if ( stop == 2)
     parmbyte |= CL_LCONT_STOP;

  if (parity != CL_PCS_PARITY_NONE)
     parmbyte |= CL_LCONT_PARITY_ENABLE;

  if (parity == CL_PCS_PARITY_EVEN)
     parmbyte |= CL_LCONT_PARITY_SELECT;

  _outp(CL_REG_LCONT, parmbyte);

}

// Default initialization is 9600 Baud, 8 data bits and 1 stop bit
void CL_InitDefaultSendMstPCS(CL_SEND_M_PCS *chan)
{
   chan->dsr_p = CL_MSTAT_DSR;
   chan->cts_p = CL_MSTAT_CTS;
   CL_PORT_ADS = CL_PCS_PORT_COM1; // set the port to be COM1 , default
   chan->txd_p = CL_REG_TX;
   baudset((long) 9600);           // 9600 Baud, 8 data bits, 1 stop bit
   comparm(CL_PCS_PARITY_NONE, CL_PCS_STOPBITS_1, CL_PCS_DATABITS_8);
   _outp(CL_REG_MCONT,0);          // Initialize outgoing handshake signals
}

// Default initialization is 9600 Baud, 8 data bits and 1 stop bit
void CL_InitDefaultRecMstPCS(CL_REC_M_PCS *chan)
{
   chan->dtr_p = CL_MCONT_DTR;
   chan->rts_p = CL_MCONT_RTS;
   CL_PORT_ADS = CL_PCS_PORT_COM1; // set the port to be COM1 , default
   chan->rxd_p = CL_REG_RX;
   baudset((long) 9600);           // 9600 Baud, 8 data bits, 1 stop bit
   comparm(CL_PCS_PARITY_NONE, CL_PCS_STOPBITS_1, CL_PCS_DATABITS_8);
   _outp(CL_REG_MCONT,0);          // Initialize outgoing handshake signals
}

// Initailze to the specified parameters
void CL_InitSendMstPCS(CL_SEND_M_PCS *chan, int port, long baud_rate, int parity,
					       int data_bits, int stop_bits)
{
   chan->dsr_p = CL_MSTAT_DSR;
   chan->cts_p = CL_MSTAT_CTS;
   if ( port == CL_PCS_PORT_COM1 ) // set the PC to the specified COM port
       CL_PORT_ADS = CL_PCS_PORT_COM1;
   else if ( port == CL_PCS_PORT_COM2 )
       CL_PORT_ADS = CL_PCS_PORT_COM2;
   else
       printf("Invalid COM port specified\n");
   chan->txd_p = CL_REG_TX;
   baudset((long) baud_rate);      // set the sepecified baud rate etc.
   comparm(parity,stop_bits,data_bits);
   _outp(CL_REG_MCONT,0);          // Initialize the handshake signals
}

// Initailze to the specified parameters
void CL_InitRecMstPCS(CL_REC_M_PCS *chan, int port, long baud_rate, int parity,
					       int data_bits, int stop_bits)
{
   chan->dtr_p = CL_MCONT_DTR;
   chan->rts_p = CL_MCONT_RTS;
   if ( port == CL_PCS_PORT_COM1 ) // set the PC to the specified COM port
       CL_PORT_ADS = CL_PCS_PORT_COM1;
   else if ( port == CL_PCS_PORT_COM2 )
       CL_PORT_ADS = CL_PCS_PORT_COM2;
   else
       printf("Invalid COM port specified\n");
   chan->rxd_p = CL_REG_RX;
   baudset((long) baud_rate);      // set the sepecified baud rate etc.
   comparm(parity,stop_bits,data_bits);
   _outp(CL_REG_MCONT,0);          // Initialize the handshake signals
}

// Send a byte to the device using handshaking
void CL_SendMstPCS(CL_SEND_M_PCS chan, uchar message)
{
   // Modem status register
   uint mstat=0;

   // First wait for the device to assert DSR, it is ready to start
   // receiving

   do
      {mstat = _inp(CL_REG_MSTAT);}
   while(!(mstat & chan.dsr_p));

   mstat=0;                        // reset mstat to 0

   _outp(chan.txd_p,message);      // send the character

   // Now wait for the device to assert CTS, it has processed the
   // message

   do
      {mstat = _inp(CL_REG_MSTAT);}
   while(!(mstat & chan.cts_p));

   mstat=0;

   //Next wait for the device to set both DSR AND CTS low, this is done on the
   //PC side so that the PC does not race ahead of slow devices
   do
      {mstat = _inp(CL_REG_MSTAT);}
   while(mstat & chan.dsr_p);
   
   mstat=0;

   do
      {mstat = _inp(CL_REG_MSTAT);}
   while(mstat & chan.cts_p);
}

// Receive a byte from the device using handshaking

uchar CL_RecMstPCS(CL_REC_M_PCS chan)
{
   uchar byte_received;

   // Line status register
   uint lstat=0;

   // Assert DTR, the primary outgoing handshake
   _outp(CL_REG_MCONT, chan.dtr_p);

   // Wait for character to be received without errors
   while (!(lstat & CL_LSTAT_DATA_READY))
   {
      lstat = _inp(CL_REG_LSTAT);

      if(lstat & CL_LSTAT_OVERRUN)
	 printf("Overrun error\n");

      if(lstat & CL_LSTAT_PARITY_ERROR)
	 printf("Parity error\n");

      if(lstat & CL_LSTAT_FRAME_ERROR)
	 printf("Frame error\n");

      if(lstat & CL_LSTAT_BREAK_INT)
	 printf("Break received\n");
   }

   // Assert RTS, the secondary outgoing handshake
   _outp(CL_REG_MCONT, chan.rts_p);
   
   delay(longd);                   // delay for slow devices, such as the 8051

   // Read the data byte
   byte_received = _inp(chan.rxd_p);

   // Clear the handshaking lines
   _outp(CL_REG_MCONT,0);

  return byte_received;
}

