-- Linus Tauro
-- FPGA protocol for the ISA bus interface (16 bits) 

library ieee;
use ieee.numeric_bit.all;

-- port definition
entity FPGA_GcdWord is
       generic(N: INTEGER := 16);
       port(iocs16_p             : inout bit;
            selectg_iocs16_p     : inout bit;   
            selectg_dataout_p    : inout bit;
            selectg_datain_p     : inout bit;
            dataout_p            : inout unsigned(N-1 downto 0); 
            datain_p             : in unsigned(N-1 downto 0);
            clk_p	         : in bit;
            address_p            : in unsigned(15 downto 0); 
            aen_p                : in bit;
            ior_p                : in bit;
            rst_p                : in bit;
            iow_p                : in bit);
end FPGA_GcdWord;

architecture AFPGA_GcdWord of FPGA_GcdWord is 

-- Using address "0300" from the unused block 0300h-0377h
constant address  : unsigned(N-1 downto 0) :="0000001100000000";  

procedure RecBlock(
            variable datain            : out unsigned(N-1 downto 0);   
            signal   recdata           : in unsigned(N-1 downto 0);
            signal   transfer_complete : in bit;
            signal   clk_p             : in bit) is

begin
      wait until clk_p'event and clk_p='1' and transfer_complete='1';      
      datain := recdata;
      wait until clk_p'event and clk_p='1' and transfer_complete='0';
end RecBlock;


procedure SendBlock(
            constant dataout           : in unsigned(N-1 downto 0);   
            signal   senddata          : out unsigned(N-1 downto 0);
            signal   transfer_complete : in bit;
            signal   clk_p             : in bit) is

begin
      senddata <= dataout;
      wait until clk_p'event and clk_p='1' and transfer_complete='1';
      wait until clk_p'event and clk_p='1' and transfer_complete='0';
end SendBlock;

procedure SendPersistent(
            constant dataout           : in unsigned(N-1 downto 0);   
            signal   senddata          : out unsigned(N-1 downto 0);
            signal   transfer_complete : in bit;
            signal   clk_p             : in bit) is

begin
      senddata <= dataout;
end SendPersistent;
            
procedure ISA_Bus_Initialize(
            signal selectg_dataout_p   : out bit;
            signal selectg_datain_p    : out bit;
            signal iocs16_p            : out bit;
            signal selectg_iocs16_p    : out bit) is

begin
     selectg_dataout_p <= '1';
     selectg_datain_p  <= '0';
     iocs16_p          <= '0';
     selectg_iocs16_p  <= '1';
end ISA_Bus_Initialize;

      signal senddata            : unsigned(N-1 downto 0);
      signal recdata             : unsigned(N-1 downto 0);
      signal transfer_complete   : bit;  

begin

 main:process

      variable x                 : unsigned(N-1 downto 0);
      variable y	         : unsigned(N-1 downto 0);     

 begin   

    --want to have RecBlock(x)
    RecBlock(x,recdata,transfer_complete,clk_p);

    --want to have RecBlock(y)
    RecBlock(y,recdata,transfer_complete,clk_p);

    SendPersistent("0000000000000000",senddata,transfer_complete,clk_p);
    
    while (x /= y) loop
       if(x < y) then
          y := y-x;
       else x:= x-y;
       end if;
    end loop;

    --want to have SendBlock(gcd_done)
    SendBlock("0000000000000001",senddata,transfer_complete,clk_p);

    --want to have SendBlock(x)
    SendBlock(x,senddata,transfer_complete,clk_p);

 end process;

 bus_control:process

 begin 

    ISA_Bus_Initialize(selectg_dataout_p,selectg_datain_p,iocs16_p,
                                                            selectg_iocs16_p);

    loop
       wait until (clk_p'event and clk_p='1' and aen_p='0' and rst_p='0' 
                          and (ior_p='0' or iow_p='0' ) and address_p=address);
    
       if iow_p='0' then      
          selectg_iocs16_p  <= '0';         
          recdata           <= datain_p;
          transfer_complete <= '1';
          wait until clk_p'event and clk_p='1' and iow_p='1';
          selectg_iocs16_p  <= '1';
          transfer_complete <= '0';
       else
           selectg_dataout_p <= '0';
           selectg_iocs16_p  <= '0';
           dataout_p         <= senddata;
           transfer_complete <= '1';  
           wait until clk_p'event and clk_p='1' and ior_p='1';     
           selectg_dataout_p <= '1';
           selectg_iocs16_p  <= '1';
           transfer_complete <= '0';
       end if;      
    end loop;

 end process;

end AFPGA_GcdWord;
