#include <stdio.h>
#include "PCSerial.h"

/******************************************************************************
PCSerial.c : PC Serial communication library using handshaking
When the device writes to the PC :
It waits for DTR to be low, sends the byte of data and then waits for RTS 
to be low. (the PC waits for a bit after asserting RTS so the device has
time to sample it.)
When the device reads from the PC :
It first sets DSR to 0, waits until it receives the byte of data and then
sets CTS to 0. The PC waits for both of them to be high before proceeding.
This avoids a race condition.
In both cases active low logic applies (from the device point of view), 
assuming a direct connection via an RS232-TTL driver/receiver chip such as
the LT1130CN (Linear Technology)
*******************************************************************************/

static int port_ads;                // holds the COM port address

static void pc_delay(int t)
{
   int i,j;
   for(i=0;i<t;++i)
      for(j=0;j<t;++j);
}

// Get a byte from the I/O port specified
static unsigned char _inp(int rec_address)
{
  unsigned char rec_byte;
  asm {
	mov dx, rec_address
	in  al,dx
	mov rec_byte,al
      }
  return rec_byte;
}

// Send a byte to the I/O port specified
static void _outp(int send_address, unsigned char send_byte)
{
  asm {
	mov dx,send_address
	mov al,send_byte
	out dx,al
      }
}

// Set the PC to the baud rate specified
static void baudset(long baudrate)
{
  unsigned int divisor;
  unsigned char lsb, msb;

  divisor = 115200l / baudrate;

  msb = divisor >> 8;
  lsb = ( divisor << 8 )  >> 8;

  _outp(REG_LCONT, LCONT_DLAB);

  _outp(port_ads, lsb);

  _outp(port_ads+1, msb);
}

// Set the parity, data bits and stop bits specified
static void comparm(int parity, int stop, int databits)
{
  BYTE parmbyte;

  parmbyte = databits - 5;

  if ( stop == 2)
     parmbyte |= LCONT_STOP;

  if (parity != PCSerial_PARITY_NONE)
     parmbyte |= LCONT_PARITY_ENABLE;

  if (parity == PCSerial_PARITY_EVEN)
     parmbyte |= LCONT_PARITY_SELECT;

  _outp(REG_LCONT, parmbyte);

}

// Default initialization is 9600 Baud, 8 data bits and 1 stop bit
void PCSerial_Bus_Init_Default(void)
{
   port_ads = PCSerial_PORT_COM1;    // set the port to be COM1 , default
   baudset((long) 9600);             // 9600 Baud, 8 data bits, 1 stop bit
   comparm(PCSerial_PARITY_NONE, PCSerial_STOPBITS_1, PCSerial_DATABITS_8);
   _outp(REG_MCONT,0);               // Initialize outgoing handshake signals
}

// Initailze to the specified parameters
void PCSerial_Bus_Init_Specify(int port, long baud_rate, int parity,
					       int data_bits, int stop_bits)
{
   if ( port == PCSerial_PORT_COM1 ) // set the PC to the specified COM port
       port_ads = PCSerial_PORT_COM1;
   else if ( port == PCSerial_PORT_COM2 )
       port_ads = PCSerial_PORT_COM2;
   else
       printf("Invalid COM port specified\n");

   baudset((long) baud_rate);        // set the sepecified baud rate etc.
   comparm(parity,stop_bits,data_bits);
   _outp(REG_MCONT,0);               // Initialize the handshake signals
}

// Send a byte to the device using handshaking
void PCSerial_PC_SendChar(unsigned char message)
{
   // Modem status register
   unsigned int mstat=0;

   // First wait for the device to assert DSR, it is ready to start
   // receiving

   do
      {mstat = _inp(REG_MSTAT);}
   while((mstat & MSTAT_DSR)==0);

   mstat=0;                         // reset mstat to 0

   _outp(REG_TX,message);           // send the character

   // Now wait for the device to assert CTS, it has processed the
   // message

   do
      {mstat = _inp(REG_MSTAT);}
   while((mstat & MSTAT_CTS)==0);

   mstat=0;

   //Next wait for the device to set both DSR AND CTS low, this is done on the
   //PC side so that the PC does not race ahead of slow devices
   do
      {mstat = _inp(REG_MSTAT);}
   while((mstat & MSTAT_DSR)!=0);

   mstat=0;

   do
      {mstat = _inp(REG_MSTAT);}
   while((mstat & MSTAT_CTS)!=0);
}

// Receive a byte from the device using handshaking

unsigned char PCSerial_PC_ReceiveChar(void)
{
   unsigned char byte_received;

   // Line status register
   unsigned int lstat=0;

   // Assert DTR, the primary outgoing handshake
   _outp(REG_MCONT, MCONT_DTR);

   // Wait for character to be received without errors
   while (!(lstat & LSTAT_DATA_READY))
   {
      lstat = _inp(REG_LSTAT);

      if(lstat & LSTAT_OVERRUN)
	 printf("Overrun error\n");

      if(lstat & LSTAT_PARITY_ERROR)
	 printf("Parity error\n");

      if(lstat & LSTAT_FRAME_ERROR)
	 printf("Frame error\n");

      if(lstat & LSTAT_BREAK_INT)
	 printf("Break received\n");
   }

   // Assert RTS, the secondary outgoing handshake
   _outp(REG_MCONT, MCONT_RTS);
   delay(longd);                   // delay for slow devices, such as the 8051

   // Read the data byte
   byte_received = _inp(REG_RX);

   // Clear the handshaking lines
   _outp(REG_MCONT,0);

  return byte_received;
}

