#ifndef cl_8051_h
#define cl_8051_h
#include "cl_decl.h"
#include <reg51.h>

/**************************************************************************** 

   CL_8051.h:
      Communication Library header file for transferring data to and from
      an 8051.

   The data transfer can be done using 8, 4 or 1 data lines and the protocol
   selected can be 4 phase (request and acknowledge lines) or 2 phase
   (only request line with a fixed wait time).
   In addition an addressed mode of communication can be selected.
   The data size is assumed to be a byte.

   The number of address lines is the same as the number of data lines (an
   8 bit address is used as well) except for the following two exceptions :
   4 data lines, 4 phase and 1 data line, 4 phase because it is not
   practical to send the address in multiple bursts in these cases, since
   there is only one ack line.
   Therefore in these 2 cases a 4 bit address has been assumed, which is
   sent over 4 address lines.
   
*****************************************************************************/

// First declare all the ports and individual port pins

#define CL_P0	0x80
#define	CL_P1	0x90
#define CL_P2	0xa0
#define CL_P3	0xb0

#define	CL_P0_0	0x80
#define	CL_P0_1	0x81
#define	CL_P0_2	0x82
#define	CL_P0_3	0x83
#define	CL_P0_4	0x84
#define	CL_P0_5	0x85
#define	CL_P0_6	0x86
#define	CL_P0_7 0x87

#define	CL_P1_0	0x90
#define	CL_P1_1	0x91
#define	CL_P1_2	0x92
#define	CL_P1_3	0x93
#define	CL_P1_4	0x94
#define	CL_P1_5	0x95
#define	CL_P1_6	0x96
#define	CL_P1_7 0x97

#define	CL_P2_0	0xa0
#define	CL_P2_1	0xa1
#define	CL_P2_2	0xa2
#define	CL_P2_3	0xa3
#define	CL_P2_4	0xa4
#define	CL_P2_5	0xa5
#define	CL_P2_6	0xa6
#define	CL_P2_7 0xa7

#define	CL_P3_0	0xb0
#define	CL_P3_1	0xb1
#define	CL_P3_2	0xb2
#define	CL_P3_3	0xb3
#define	CL_P3_4	0xb4
#define	CL_P3_5	0xb5
#define	CL_P3_6	0xb6
#define	CL_P3_7 0xb7

#define	uint	unsigned int
#define	uchar	unsigned char
#define ushort	unsigned short

/* FIX THIS add these fields later 
      //int               timeout; 

/****************************************************************************** 
    8-bit data bus, 4-phase handshake (D8_P4) (request and acknowledge lines) 
******************************************************************************/ 

#if CL_SEND_M_D8P4

/* Master send -- servant receive */

typedef struct _CL_SEND_M_D8_P4_struct { 
   uchar data_p, req_p, ack_p;   
}
CL_SEND_M_D8_P4;

/* Default ports declared in the library are used for the channel */
void 	CL_InitDefaultSendMstD8P4 (CL_SEND_M_D8_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD8P4 (CL_SEND_M_D8_P4 *chan, uchar data_p_address,
                              uchar req_p_address, uchar ack_p_address);

void 	CL_SendMstD8P4 (CL_SEND_M_D8_P4 chan, uchar char_to_send);

#endif

#if CL_REC_S_D8P4

typedef struct _CL_REC_S_D8_P4_struct { 
   uchar data_p, req_p, ack_p;
} 
CL_REC_S_D8_P4;
void 	CL_InitDefaultRecSrvD8P4 (CL_REC_S_D8_P4 *chan);
void    CL_InitRecSrvD8P4 (CL_REC_S_D8_P4 *chan,  uchar data_p_address,
                            uchar req_p_address,  uchar ack_p_address);
uchar	CL_RecSrvD8P4 (CL_REC_S_D8_P4 chan);

#endif

#if CL_REC_M_D8P4

/* Master receive -- servant send */

typedef struct _CL_REC_M_D8_P4_struct { 
   uchar data_p, req_p, ack_p;   
}
CL_REC_M_D8_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD8P4 (CL_REC_M_D8_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD8P4 (CL_REC_M_D8_P4 *chan,  uchar data_p_address,
                            uchar req_p_address,  uchar ack_p_address);

uchar	CL_RecMstD8P4 (CL_REC_M_D8_P4 chan);

#endif

#if CL_SEND_S_D8P4

typedef struct _CL_SEND_S_D8_P4_struct { 
   uchar data_p, req_p, ack_p;   
}
CL_SEND_S_D8_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD8P4 (CL_SEND_S_D8_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD8P4 (CL_SEND_S_D8_P4 *chan,  uchar data_p_address,
                              uchar req_p_address,  uchar ack_p_address);

void	CL_SendSrvD8P4 (CL_SEND_S_D8_P4 chan, uchar char_to_send);

#endif

#if CL_SEND_M_D8P4_A

/* Master send --  servant receive,  addressed */

typedef struct _CL_SEND_M_D8_P4_A_struct { 
   uchar data_p, req_p, ack_p, addr_p, addr_req_p;   
}
CL_SEND_M_D8_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD8P4A (CL_SEND_M_D8_P4_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD8P4A(CL_SEND_M_D8_P4_A *chan,  uchar data_p_address,
                               uchar req_p_address,  uchar ack_p_address,
                               uchar addr_p_address, uchar addr_req_p_address);

void 	CL_SendMstD8P4A
           	(CL_SEND_M_D8_P4_A chan, uchar addr, uchar char_to_send);

#endif

#if CL_REC_S_D8P4_A

typedef struct _CL_REC_S_D8_P4_A_struct { 
   uchar data_p, req_p, ack_p, addr_p, addr_req_p;   
   uchar addr;  // the address of this receiver
}
CL_REC_S_D8_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD8P4A (CL_REC_S_D8_P4_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD8P4A (CL_REC_S_D8_P4_A *chan,  uchar data_p_address,
        uchar req_p_address,  uchar ack_p_address,   uchar addr_p_address,
                              uchar addr_req_p_address, uchar addr);

uchar	CL_RecSrvD8P4A (CL_REC_S_D8_P4_A chan);

#endif

#if CL_REC_M_D8P4_A

/* Master receive --  servant send,  addressed */

typedef struct _CL_REC_M_D8_P4_A_struct { 
   uchar data_p, req_p, ack_p, addr_req_p, addr_p;   
}
CL_REC_M_D8_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD8P4A (CL_REC_M_D8_P4_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD8P4A(CL_REC_M_D8_P4_A *chan,  uchar data_p_address,
                               uchar req_p_address,  uchar ack_p_address,
                               uchar addr_p_address, uchar addr_req_p_address);

uchar 	CL_RecMstD8P4A (CL_REC_M_D8_P4_A chan, uchar addr);

#endif

#if CL_SEND_S_D8P4_A

typedef struct _CL_SEND_S_D8_P4_A_struct { 
   uchar data_p, req_p, ack_p, addr_req_p, addr_p;   
   uchar addr;   // the address of the receiver
}
CL_SEND_S_D8_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD8P4A (CL_SEND_S_D8_P4_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD8P4A (CL_SEND_S_D8_P4_A *chan,  uchar data_p_address,
                           uchar req_p_address,  uchar ack_p_address,
              uchar addr_p_address, uchar addr_req_p_address, uchar addr);

void    CL_SendSrvD8P4A (CL_SEND_S_D8_P4_A chan, uchar char_to_send);

#endif


/****************************************************************************** 
    4-bit data bus, 4-phase handshake (D4_P4) (request and acknowledge lines) 
******************************************************************************/ 

#if CL_SEND_M_D4P4

/* Master send -- servant receive */

typedef struct _CL_SEND_M_D4_P4_struct { 
   uchar data4_p;                
   uchar req_p, ack_p;
}
CL_SEND_M_D4_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD4P4 (CL_SEND_M_D4_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD4P4 (CL_SEND_M_D4_P4 *chan,  uchar data4_p_base_address,
                           uchar req_p_address, uchar ack_p_address);

void    CL_SendMstD4P4 (CL_SEND_M_D4_P4 chan, uchar char_to_send);

#endif

#if CL_REC_S_D4P4

typedef struct _CL_REC_S_D4_P4_struct { 
   uchar data4_p;
   uchar req_p, ack_p;
} 
CL_REC_S_D4_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD4P4 (CL_REC_S_D4_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD4P4 (CL_REC_S_D4_P4 *chan,  uchar data4_p_base_address,
                           uchar req_p_address, uchar ack_p_address);

uchar   CL_RecSrvD4P4 (CL_REC_S_D4_P4 chan);

#endif

#if CL_REC_M_D4P4

/* Master receive -- servant send */

typedef struct _CL_REC_M_D4_P4_struct { 
   uchar data4_p; 
   uchar req_p, ack_p;
}
CL_REC_M_D4_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD4P4 (CL_REC_M_D4_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD4P4 (CL_REC_M_D4_P4 *chan,  uchar data4_p_base_address,
                           uchar req_p_address, uchar ack_p_address);

uchar   CL_RecMstD4P4 (CL_REC_M_D4_P4 chan);

#endif

#if CL_SEND_S_D4P4

typedef struct _CL_SEND_S_D4_P4_struct { 
   uchar data4_p;
   uchar req_p, ack_p;
}
CL_SEND_S_D4_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD4P4 (CL_SEND_S_D4_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD4P4 (CL_SEND_S_D4_P4 *chan,  uchar data4_p_base_address,
                           uchar req_p_address, uchar ack_p_address);

void    CL_SendSrvD4P4 (CL_SEND_S_D4_P4 chan, uchar char_to_send);

#endif

#if CL_SEND_M_D4P4_A

/* Master send --  servant receive,  addressed */

typedef struct _CL_SEND_M_D4_P4_A_struct {
   uchar data4_p, req_p, ack_p; 
   uchar addr4_p, addr_req_p;
}
CL_SEND_M_D4_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD4P4A (CL_SEND_M_D4_P4_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD4P4A (CL_SEND_M_D4_P4_A *chan,  uchar data4_p_base_address,
                             uchar req_p_address, uchar ack_p_address,
                      uchar addr4_p_base_address, uchar addr_req_p_address);

void    CL_SendMstD4P4A
                (CL_SEND_M_D4_P4_A chan, uchar addr, uchar char_to_send);

#endif

#if CL_REC_S_D4P4_A

typedef struct _CL_REC_S_D4_P4_A_struct { 
   uchar data4_p, req_p, ack_p;
   uchar addr4_p, addr_req_p;
   uchar addr;  // the address of this receiver
}
CL_REC_S_D4_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD4P4A (CL_REC_S_D4_P4_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD4P4A (CL_REC_S_D4_P4_A *chan,  uchar data4_p_base_address,
        uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                                uchar addr_req_p_address, uchar addr);

uchar   CL_RecSrvD4P4A (CL_REC_S_D4_P4_A chan);

#endif

#if CL_REC_M_D4P4_A

/* Master receive --  servant send,  addressed */

typedef struct _CL_REC_M_D4_P4_A_struct { 
   uchar data4_p, req_p, ack_p;
   uchar addr4_p, addr_req_p;
}
CL_REC_M_D4_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD4P4A (CL_REC_M_D4_P4_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD4P4A (CL_REC_M_D4_P4_A *chan,  uchar data4_p_base_address,
        uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                                                   uchar addr_req_p_address); 

uchar   CL_RecMstD4P4A (CL_REC_M_D4_P4_A chan, uchar addr);

#endif

#if CL_SEND_S_D4P4_A

typedef struct _CL_SEND_S_D4_P4_A_struct { 
   uchar data4_p, req_p, ack_p;
   uchar addr4_p, addr_req_p;
   uchar addr;   // the address of the receiver
}
CL_SEND_S_D4_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD4P4A (CL_SEND_S_D4_P4_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD4P4A (CL_SEND_S_D4_P4_A *chan,  uchar data4_p_base_address,
        uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                                uchar addr_req_p_address, uchar addr);

void    CL_SendSrvD4P4A (CL_SEND_S_D4_P4_A chan, uchar char_to_send);

#endif

/****************************************************************************** 
    1-bit data bus, 4-phase handshake (D1_P4) (request and acknowledge lines) 
******************************************************************************/ 

#if CL_SEND_M_D1P4

/* Master send -- servant receive */

typedef struct _CL_SEND_M_D1_P4_struct { 
   uchar data1_p, req_p, ack_p;   
}
CL_SEND_M_D1_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD1P4 (CL_SEND_M_D1_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD1P4 (CL_SEND_M_D1_P4 *chan, uchar data1_p_address,
                               uchar req_p_address, uchar ack_p_address);

void    CL_SendMstD1P4 (CL_SEND_M_D1_P4 chan, uchar char_to_send);

#endif

#if CL_REC_S_D1P4

typedef struct _CL_REC_S_D1_P4_struct { 
   uchar data1_p, req_p, ack_p;
} 
CL_REC_S_D1_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD1P4 (CL_REC_S_D1_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD1P4 (CL_REC_S_D1_P4 *chan, uchar data1_p_address,
                               uchar req_p_address, uchar ack_p_address);

uchar   CL_RecSrvD1P4 (CL_REC_S_D1_P4 chan);

#endif

#if CL_REC_M_D1P4

/* Master receive -- servant send */

typedef struct _CL_REC_M_D1_P4_struct { 
   uchar data1_p, req_p, ack_p;   
}
CL_REC_M_D1_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD1P4 (CL_REC_M_D1_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD1P4 (CL_REC_M_D1_P4 *chan, uchar data1_p_address,
                               uchar req_p_address, uchar ack_p_address);

uchar   CL_RecMstD1P4 (CL_REC_M_D1_P4 chan);

#endif

#if CL_SEND_S_D1P4

typedef struct _CL_SEND_S_D1_P4_struct { 
   uchar data1_p, req_p, ack_p;   
}
CL_SEND_S_D1_P4;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD1P4 (CL_SEND_S_D1_P4 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD1P4 (CL_SEND_S_D1_P4 *chan, uchar data1_p_address,
                               uchar req_p_address, uchar ack_p_address);

void    CL_SendSrvD1P4 (CL_SEND_S_D1_P4 chan, uchar char_to_send);

#endif

#if CL_SEND_M_D1P4_A

/* Master send --  servant receive,  addressed */

typedef struct _CL_SEND_M_D1_P4_A_struct { 
   uchar data1_p, req_p, ack_p;
   uchar addr4_p, addr_req_p;
}
CL_SEND_M_D1_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD1P4A (CL_SEND_M_D1_P4_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD1P4A (CL_SEND_M_D1_P4_A *chan, uchar data1_p_address,
        uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                                                  uchar addr_req_p_address);

void    CL_SendMstD1P4A
                (CL_SEND_M_D1_P4_A chan, uchar addr, uchar char_to_send);

#endif

#if CL_REC_S_D1P4_A

typedef struct _CL_REC_S_D1_P4_A_struct { 
   uchar data1_p, req_p, ack_p;
   uchar addr4_p, addr_req_p;
   uchar addr;  // the address of this receiver
}
CL_REC_S_D1_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD1P4A (CL_REC_S_D1_P4_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD1P4A (CL_REC_S_D1_P4_A *chan, uchar data1_p_address,
        uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                                 uchar addr_req_p_address, uchar addr);

uchar   CL_RecSrvD1P4A (CL_REC_S_D1_P4_A chan);

#endif

#if CL_REC_M_D1P4_A

/* Master receive --  servant send,  addressed */

typedef struct _CL_REC_M_D1_P4_A_struct { 
   uchar data1_p, req_p, ack_p;
   uchar addr4_p, addr_req_p;
}
CL_REC_M_D1_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD1P4A (CL_REC_M_D1_P4_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD1P4A (CL_REC_M_D1_P4_A *chan, uchar data1_p_address,
        uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                                                  uchar addr_req_p_address);

uchar   CL_RecMstD1P4A (CL_REC_M_D1_P4_A chan, uchar addr);

#endif

#if CL_SEND_S_D1P4_A

typedef struct _CL_SEND_S_D1_P4_A_struct { 
   uchar data1_p, req_p, ack_p;
   uchar addr4_p, addr_req_p;
   uchar addr;   // the address of the receiver
}
CL_SEND_S_D1_P4_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD1P4A (CL_SEND_S_D1_P4_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD1P4A (CL_SEND_S_D1_P4_A *chan, uchar data1_p_address,
        uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                                   uchar addr_req_p_address, uchar addr);

void    CL_SendSrvD1P4A (CL_SEND_S_D1_P4_A chan, uchar char_to_send);

#endif

/****************************************************************************** 
    8-bit data bus, 2-phase handshake (D8_P2) (req line only, fixed wait time) 
******************************************************************************/ 

#if CL_SEND_M_D8P2

/* Master send -- servant receive */

typedef struct _CL_SEND_M_D8_P2_struct { 
   uchar data_p, req_p;
   uint  wait_time;
   uint  oneus_loop;
}
CL_SEND_M_D8_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD8P2 (CL_SEND_M_D8_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD8P2 (CL_SEND_M_D8_P2 *chan, uchar data_p_address,
                  uchar req_p_address, uint wait_time, uint oneus_loop);

void    CL_SendMstD8P2 (CL_SEND_M_D8_P2 chan, uchar char_to_send);

#endif

#if CL_REC_S_D8P2

typedef struct _CL_REC_S_D8_P2_struct { 
   uchar data_p, req_p;   
} 
CL_REC_S_D8_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD8P2 (CL_REC_S_D8_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD8P2 (CL_REC_S_D8_P2 *chan, uchar data_p_address,
                                                    uchar req_p_address);

uchar   CL_RecSrvD8P2 (CL_REC_S_D8_P2 chan);

#endif

#if CL_REC_M_D8P2

/* Master receive -- servant send */

typedef struct _CL_REC_M_D8_P2_struct { 
   uchar data_p, req_p;   
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_REC_M_D8_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD8P2 (CL_REC_M_D8_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD8P2 (CL_REC_M_D8_P2 *chan, uchar data_p_address,
                  uchar req_p_address, uint wait_time, uint oneus_loop);

uchar   CL_RecMstD8P2 (CL_REC_M_D8_P2 chan);

#endif

#if CL_SEND_S_D8P2

typedef struct _CL_SEND_S_D8_P2_struct { 
   uchar data_p, req_p;   
} 
CL_SEND_S_D8_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD8P2 (CL_SEND_S_D8_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD8P2 (CL_SEND_S_D8_P2 *chan, uchar data_p_address,
                                                     uchar req_p_address);

void    CL_SendSrvD8P2 (CL_SEND_S_D8_P2 chan, uchar char_to_send);

#endif

#if CL_SEND_M_D8P2_A

/* Master send -- servant receive, addressed */

typedef struct _CL_SEND_M_D8_P2_A_struct { 
   uchar data_p, req_p, addr_p, addr_req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_SEND_M_D8_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD8P2A (CL_SEND_M_D8_P2_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD8P2A (CL_SEND_M_D8_P2_A *chan, uchar data_p_address,
       uchar req_p_address, uchar addr_p_address, uchar addr_req_p_address,
                            uint  wait_time,  uint oneus_loop);

void    CL_SendMstD8P2A (CL_SEND_M_D8_P2_A chan, uchar addr, uchar char_to_send);

#endif

#if CL_REC_S_D8P2_A

typedef struct _CL_REC_S_D8_P2_A_struct { 
   uchar data_p, addr_p, req_p, addr_req_p;
   uchar addr;                      // address of this receiver
} 
CL_REC_S_D8_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD8P2A (CL_REC_S_D8_P2_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD8P2A (CL_REC_S_D8_P2_A *chan, uchar data_p_address,
       uchar req_p_address, uchar addr_p_address, uchar addr_req_p_address,
                                                      uchar addr);

uchar   CL_RecSrvD8P2A (CL_REC_S_D8_P2_A chan);

#endif

#if CL_REC_M_D8P2_A

/* Master receive -- servant send, addressed */

typedef struct _CL_REC_M_D8_P2_A_struct { 
   uchar data_p, addr_p, req_p, addr_req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_REC_M_D8_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD8P2A (CL_REC_M_D8_P2_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD8P2A (CL_REC_M_D8_P2_A *chan, uchar data_p_address,
       uchar req_p_address, uchar addr_p_address, uchar addr_req_p_address,
                            uint  wait_time,  uint oneus_loop);

uchar   CL_RecMstD8P2A (CL_REC_M_D8_P2_A chan, uchar addr);

#endif

#if CL_SEND_S_D8P2_A

typedef struct _CL_SEND_S_D8_P2_A_struct { 
   uchar data_p, req_p, addr_p, addr_req_p;
   uchar addr;                     // address of the receiver
} 
CL_SEND_S_D8_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD8P2A (CL_SEND_S_D8_P2_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD8P2A (CL_SEND_S_D8_P2_A *chan, uchar data_p_address,
       uchar req_p_address, uchar addr_p_address, uchar addr_req_p_address,
                                                      uchar addr);

void    CL_SendSrvD8P2A (CL_SEND_S_D8_P2_A chan, uchar char_to_send);

#endif

/****************************************************************************** 
    4-bit data bus, 2-phase handshake (D4_P2) (req line only, fixed wait time) 
******************************************************************************/ 

#if CL_SEND_M_D4P2

/* Master send -- servant receive */

typedef struct _CL_SEND_M_D4_P2_struct { 
   uchar data4_p, req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_SEND_M_D4_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD4P2 (CL_SEND_M_D4_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD4P2 (CL_SEND_M_D4_P2 *chan, uchar data4_p_base_address,
                uchar req_p_address, uint wait_time, uint oneus_loop);

void    CL_SendMstD4P2 (CL_SEND_M_D4_P2 chan, uchar char_to_send);

#endif

#if CL_REC_S_D4P2

typedef struct _CL_REC_S_D4_P2_struct {
   uchar data4_p, req_p;
} 
CL_REC_S_D4_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD4P2 (CL_REC_S_D4_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD4P2 (CL_REC_S_D4_P2 *chan, uchar data4_p_base_address,
                                                      uchar req_p_address);

uchar   CL_RecSrvD4P2 (CL_REC_S_D4_P2 chan);

#endif

#if CL_REC_M_D4P2

/* Master receive -- servant send */

typedef struct _CL_REC_M_D4_P2_struct { 
   uchar data4_p, req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_REC_M_D4_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD4P2 (CL_REC_M_D4_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD4P2 (CL_REC_M_D4_P2 *chan, uchar data4_p_base_address,
                uchar req_p_address, uint wait_time, uint oneus_loop);

uchar   CL_RecMstD4P2 (CL_REC_M_D4_P2 chan);

#endif

#if CL_SEND_S_D4P2

typedef struct _CL_SEND_S_D4_P2_struct {
   uchar data4_p, req_p;
} 
CL_SEND_S_D4_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD4P2 (CL_SEND_S_D4_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD4P2 (CL_SEND_S_D4_P2 *chan, uchar data4_p_base_address,
                                                uchar req_p_address);

void    CL_SendSrvD4P2 (CL_SEND_S_D4_P2 chan, uchar char_to_send);

#endif

#if CL_SEND_M_D4P2_A

/* Master send -- servant receive, addressed */

typedef struct _CL_SEND_M_D4_P2_A_struct {
   uchar data4_p, req_p;
   uchar addr4_p, addr_req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_SEND_M_D4_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD4P2A (CL_SEND_M_D4_P2_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD4P2A (CL_SEND_M_D4_P2_A *chan, uchar data4_p_base_address,
    uchar req_p_address, uchar addr4_p_base_address, uchar addr_req_p_address,
                                            uint wait_time, uint oneus_loop);

void    CL_SendMstD4P2A (CL_SEND_M_D4_P2_A chan, uchar addr, uchar char_to_send);

#endif

#if CL_REC_S_D4P2_A

typedef struct _CL_REC_S_D4_P2_A_struct {
   uchar data4_p, req_p;
   uchar addr4_p, addr_req_p;
   uchar addr;                      // address of this receiver
} 
CL_REC_S_D4_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD4P2A (CL_REC_S_D4_P2_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD4P2A (CL_REC_S_D4_P2_A *chan, uchar data4_p_base_address,
    uchar req_p_address, uchar addr4_p_base_address, uchar addr_req_p_address,
                                                           uchar addr);

uchar   CL_RecSrvD4P2A (CL_REC_S_D4_P2_A chan);

#endif

#if CL_REC_M_D4P2_A

/* Master receive -- servant send, addressed */

typedef struct _CL_REC_M_D4_P2_A_struct {
   uchar data4_p, req_p;
   uchar addr4_p, addr_req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_REC_M_D4_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD4P2A (CL_REC_M_D4_P2_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD4P2A (CL_REC_M_D4_P2_A *chan, uchar data4_p_base_address,
    uchar req_p_address, uchar addr4_p_base_address, uchar addr_req_p_address,
                                            uint wait_time, uint oneus_loop);

uchar   CL_RecMstD4P2A (CL_REC_M_D4_P2_A chan, uchar addr);

#endif

#if CL_SEND_S_D4P2_A

typedef struct _CL_SEND_S_D4_P2_A_struct { 
   uchar data4_p, req_p;
   uchar addr4_p, addr_req_p;
   uchar addr;                     // address of the receiver
} 
CL_SEND_S_D4_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD4P2A (CL_SEND_S_D4_P2_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD4P2A (CL_SEND_S_D4_P2_A *chan, uchar data4_p_base_address,
    uchar req_p_address, uchar addr4_p_base_address, uchar addr_req_p_address,
                                                      uchar addr);

void    CL_SendSrvD4P2A (CL_SEND_S_D4_P2_A chan, uchar char_to_send);

#endif

/****************************************************************************** 
    1-bit data bus, 2-phase handshake (D1_P2) (req line only, fixed wait time) 
******************************************************************************/ 

#if CL_SEND_M_D1P2

/* Master send -- servant receive */

typedef struct _CL_SEND_M_D1_P2_struct { 

   uchar data1_p, req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_SEND_M_D1_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD1P2 (CL_SEND_M_D1_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD1P2 (CL_SEND_M_D1_P2 *chan, uchar data1_p_address,
       uchar req_p_address, uint wait_time, uint oneus_loop);

void    CL_SendMstD1P2 (CL_SEND_M_D1_P2 chan, uchar char_to_send);

#endif

#if CL_REC_S_D1P2

typedef struct _CL_REC_S_D1_P2_struct { 
   uchar data1_p, req_p;
} 
CL_REC_S_D1_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD1P2 (CL_REC_S_D1_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD1P2 (CL_REC_S_D1_P2 *chan, uchar data1_p_address,
                                                 uchar req_p_address); 

uchar   CL_RecSrvD1P2 (CL_REC_S_D1_P2 chan);

#endif

#if CL_REC_M_D1P2

/* Master receive -- servant send */

typedef struct _CL_REC_M_D1_P2_struct { 
   uchar data1_p, req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_REC_M_D1_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD1P2 (CL_REC_M_D1_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD1P2 (CL_REC_M_D1_P2 *chan, uchar data1_p_address,
       uchar req_p_address, uint wait_time, uint oneus_loop);

uchar   CL_RecMstD1P2 (CL_REC_M_D1_P2 chan);

#endif

#if CL_SEND_S_D1P2

typedef struct _CL_SEND_S_D1_P2_struct { 
   uchar data1_p, req_p;
} 
CL_SEND_S_D1_P2;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD1P2 (CL_SEND_S_D1_P2 *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD1P2 (CL_SEND_S_D1_P2 *chan, uchar data1_p_address,
                                               uchar req_p_address);

void    CL_SendSrvD1P2 (CL_SEND_S_D1_P2 chan, uchar char_to_send);

#endif

#if CL_SEND_M_D1P2_A

/* Master send -- servant receive, addressed */

typedef struct _CL_SEND_M_D1_P2_A_struct { 
   uchar data1_p, req_p, addr1_p, addr_req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_SEND_M_D1_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendMstD1P2A (CL_SEND_M_D1_P2_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendMstD1P2A (CL_SEND_M_D1_P2_A *chan, uchar data1_p_address,
    uchar req_p_address, uchar addr1_p_address, uchar addr_req_p_address,
                            uint wait_time, uint oneus_loop);

void    CL_SendMstD1P2A (CL_SEND_M_D1_P2_A chan, uchar addr, uchar char_to_send);

#endif

#if CL_REC_S_D1P2_A

typedef struct _CL_REC_S_D1_P2_A_struct { 
   uchar data1_p, req_p, addr1_p, addr_req_p;
   uchar addr;                      // address of this receiver
} 
CL_REC_S_D1_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecSrvD1P2A (CL_REC_S_D1_P2_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecSrvD1P2A (CL_REC_S_D1_P2_A *chan, uchar data1_p_address,
    uchar req_p_address, uchar addr1_p_address, uchar addr_req_p_address,
                                                   uchar addr);

uchar   CL_RecSrvD1P2A (CL_REC_S_D1_P2_A chan);

#endif

#if CL_REC_M_D1P2_A

/* Master receive -- servant send, addressed */

typedef struct _CL_REC_M_D1_P2_A_struct { 
   uchar data1_p, req_p, addr1_p, addr_req_p;
   uint  wait_time; 
   uint  oneus_loop; 
}
CL_REC_M_D1_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultRecMstD1P2A (CL_REC_M_D1_P2_A *chan);

/* The corresponding port addresses are passed as parameters */
void    CL_InitRecMstD1P2A (CL_REC_M_D1_P2_A *chan, uchar data1_p_address,
    uchar req_p_address, uchar addr1_p_address, uchar addr_req_p_address,
                            uint wait_time, uint oneus_loop);

uchar   CL_RecMstD1P2A (CL_REC_M_D1_P2_A chan, uchar addr);

#endif

#if CL_SEND_S_D1P2_A

typedef struct _CL_SEND_S_D1_P2_A_struct { 
   uchar data1_p, addr1_p, req_p, addr_req_p;
   uchar addr;                     // address of the receiver
} 
CL_SEND_S_D1_P2_A;

/* Default ports declared in the library are used for the channel */
void    CL_InitDefaultSendSrvD1P2A (CL_SEND_S_D1_P2_A *chan, uchar addr);

/* The corresponding port addresses are passed as parameters */
void    CL_InitSendSrvD1P2A (CL_SEND_S_D1_P2_A *chan, uchar data1_p_address,
    uchar req_p_address, uchar addr1_p_address, uchar addr_req_p_address,
                                                   uchar addr);

void    CL_SendSrvD1P2A (CL_SEND_S_D1_P2_A chan, uchar char_to_send);

#endif

/* This approach is a temporary solution to the fact that the 8051 micro-
   controller does not allow indirect addressing of the I/O ports
   Hopefully in the future the overhead associated with these procedures
   can be done away with, once indirect addressing is allowed  */

void  _CL_SetPortByte(uchar address, uchar value);
uchar _CL_GetPortByte(uchar address);
void _CL_SetPortBit(uchar address, bit value);
bit _CL_GetPortBit(uchar address);


/****************************************************************************** 
    PC-Serial (PCS) communications using the 8051 UART
******************************************************************************/ 

#if CL_S_PCSERIAL

// Define SMOD (not defined in <reg51.h>)
sbit SMOD = 0x87 ^ 7;

typedef struct _CL_SEND_S_PCS_struct
{
   uchar dsr_p, dtr_p, cts_p, rts_p;
} CL_SEND_S_PCS;

typedef struct _CL_REC_S_PCS_struct
{
   uchar dsr_p, dtr_p, cts_p, rts_p;
} CL_REC_S_PCS;

/* Always servant whether sending or receiving. */

/* Srv send */
void 	CL_InitDefaultSendSrvPCS
		(CL_SEND_S_PCS *chan, float freq);

void    CL_InitSendSrvPCS(CL_SEND_S_PCS *chan, dsr_p_address, dtr_p_address,
   cts_p_address, rts_p_address, uint baudrate, float freq); 

void CL_SendSrvPCS(CL_SEND_S_PCS chan, uchar message);

/* Servant receive */
void 	CL_InitDefaultRecSrvPCS(CL_REC_S_PCS *chan, float freq);

void    CL_InitRecSrvPCS(CL_REC_S_PCS *chan, dsr_p_address, dtr_p_address,
   cts_p_address, rts_p_address, uint baudrate, float freq); 

uchar    CL_RecSrvPCS(CL_REC_S_PCS chan);

// Errors 
#define CL_PCS_PARITY_ERROR            	1
#define CL_PCS_FRAME_ERROR             	2
#define CL_PCS_BREAK_ERROR             	3

// Allowable values for "baud_rate"
#define CL_PCS_BAUD_57600   		57600
#define CL_PCS_BAUD_38400   		38400
#define CL_PCS_BAUD_19200   		19200 
#define CL_PCS_BAUD_9600     		9600
#define CL_PCS_BAUD_4800     		4800
#define CL_PCS_BAUD_2400     		2400
#define CL_PCS_BAUD_1200     		1200
#define CL_PCS_BAUD_600       		600
#define CL_PCS_BAUD_300       		300
#define CL_PCS_BAUD_150       		150

// Allowable values for "parity"
#define CL_PCS_PARITY_NONE   		0
#define CL_PCS_PARITY_ODD    		1
#define CL_PCS_PARITY_EVEN   		2

// Allowable values for "data_bits"
#define CL_PCS_DATABITS_8    		8
#define CL_PCS_DATABITS_7    		7

// Allowable values for "stop_bits"
#define CL_PCS_STOPBITS_2    		2
#define CL_PCS_STOPBITS_1    		1

#endif

#endif


