-- FPGA protocol for the ISA bus interface (8 bits) 
-- Setinel Approach
-- Assumptions :
-- 1. The "-" operator has been overlaoded to support bit vectors. 
-- 2. Resolution functions have been written for data_p, data8_dir_p,
--    data8_g_p, cntrl_g_p

use work.cl_types.all;
use work.CL_ISA.all;

-- port definition
entity CL_S_ISA is
       generic(N: INTEGER := 8);
       port(--The following signals are common to both input and output
            addr_p               : in bit16;
            aen_p                : in bit;
            rst_p                : in bit;
            clk_p                : in bit; 
            data_p               : inout bit8;
            -- Control signals to buffer logic
            data8_g_p            : out bit; 
            data8_dir_p          : out bit;
            cntrl_g_p            : out bit;   
            -- Input channels only
            iow_p                : in bit;
            -- Output channels only
            ior_p                : in bit);
end CL_S_ISA;

architecture ACL_S_ISA of CL_S_ISA is 

-- Using addresses from the unused block 0300h-0377h
constant gcd_send_addr           : bit16 := "0000001100000000";  
constant gcd_rec_addr            : bit16 := "0000001100000001";

-- Declare the send and receive channels
signal send_chan_s               : CL_SEND_S_ISA;
signal rec_chan_s                : CL_REC_S_ISA;
signal gcd_done_chan_s           : CL_SEND_S_ISA;

signal gcd_done_s                : bit := '0';

begin
   gcd:process
      variable x                 : bit8;
      variable y	         : bit8;     
   begin   
      -- Initializations
      CL_InitSendSrvISA(send_chan_s, gcd_send_addr, data8_g_p, data8_dir_p,
                                                           cntrl_g_p, clk_p);
      CL_InitRecSrvISA(rec_chan_s, gcd_rec_addr, data8_g_p, data8_dir_p, 
                                                           cntrl_g_p, clk_p);
      
      loop
         gcd_done_s <= '0';

         -- Receive x    
         CL_RecSrvISA(rec_chan_s, x, addr_p, data_p, data8_g_p, aen_p,
                                                         rst_p, iow_p, clk_p); 
         -- Receive y
         CL_RecSrvISA(rec_chan_s, y, addr_p, data_p, data8_g_p, aen_p,
                                                         rst_p, iow_p, clk_p);
         
         -- Compute the gcd 
         while (x /= y) loop
            if(x < y) then
               y := y-x;
            else x:= x-y;
            end if;
         end loop;

         -- Set the gcd flag  
         gcd_done_s  <= '1';

         -- vcom warns of infinite loop without a wait in process, inlining
         -- should mean that the following wait is not needed
         wait until clk_p'event and clk_p='1';
               
         -- Send the result
         CL_SendSrvISA(send_chan_s, x, addr_p, data_p, data8_g_p, data8_dir_p,
                                                 aen_p, rst_p, ior_p, clk_p);
      end loop;
   end process;

   sentinel:process
   begin
      -- Initialization   
      CL_InitSendSrvISA(gcd_done_chan_s, gcd_send_addr, data8_g_p,
                                         data8_dir_p, cntrl_g_p, clk_p);

      loop
         -- Tell the ISA master that the gcd result is not computed yet 
         while (gcd_done_s = '0') loop
            CL_SendSrvISA(gcd_done_chan_s, "00000000", addr_p, data_p, 
                         data8_g_p, data8_dir_p, aen_p, rst_p, ior_p, clk_p);  
         end loop;
         wait until clk_p'event and clk_p='1' and gcd_done_s='0';
      end loop;
   end process;

end ACL_S_ISA;
  