#pragma SMALL
#pragma db   
#pragma CODE
#pragma OPTIMIZE(6, SIZE)

#include "cl_8051.h" 

/**************************************************************************** 

   CL_8051.c:
      Communication Library for transferring data to and from an 8051.

   The data transfer can be done using 8, 4 or 1 data lines and the protocol
   selected can be 4 phase (request and acknowledge lines) or 2 phase
   (only request line with a fixed wait time).
   In addition an addressed mode of communication can be selected.
   The data size is assumed to be a byte.

   The number of address lines is the same as the number of data lines (an
   8 bit address is used as well) except for the following two exceptions :
   4 data lines, 4 phase and 1 data line, 4 phase because it is not
   practical to send the address in multiple bursts in these cases, since
   there is only one ack line.
   Therefore in these 2 cases a 4 bit address has been assumed, which is
   sent over 4 address lines.
   
*****************************************************************************/

/* NOTE: Apparently, the compilers only dinstinguishes the first
   32 characters of a function name.
*/

/* NOTE: ports (and all special-function registers) can only be accessed 
   using direct addressing.  Using indirect addressing refers to the
   upper 128 bytes of internal memory.  For example, directly addressing
   0x80 refers to P0, but indirectly addressing 0x80 refers to a byte in
   internal memory.  This is very non-intuitive, but the 8051 uses this
   method to allow an 8-bit address refer to 384 bytes instead of 256.
   For us, this means that we can't store the port address in a channel
   structure and then use indirect addressing.  Instead, we have to
   use a function that looks at the address, and then branches to a statement
   that uses direct addressing to access that port.
*/


/* These decls are used by the SetPortBit and GetPortBit functions */
sbit _CL_P0_0          = P0 ^ 0;
sbit _CL_P0_1          = P0 ^ 1;
sbit _CL_P0_2          = P0 ^ 2;
sbit _CL_P0_3          = P0 ^ 3;
sbit _CL_P0_4          = P0 ^ 4;
sbit _CL_P0_5          = P0 ^ 5;
sbit _CL_P0_6          = P0 ^ 6;
sbit _CL_P0_7          = P0 ^ 7;

sbit _CL_P1_0          = P1 ^ 0;
sbit _CL_P1_1          = P1 ^ 1;
sbit _CL_P1_2          = P1 ^ 2;
sbit _CL_P1_3          = P1 ^ 3;
sbit _CL_P1_4          = P1 ^ 4;
sbit _CL_P1_5          = P1 ^ 5;
sbit _CL_P1_6          = P1 ^ 6;
sbit _CL_P1_7          = P1 ^ 7;
 
sbit _CL_P2_0          = P2 ^ 0;
sbit _CL_P2_1          = P2 ^ 1;
sbit _CL_P2_2          = P2 ^ 2;
sbit _CL_P2_3          = P2 ^ 3;
sbit _CL_P2_4          = P2 ^ 4;
sbit _CL_P2_5          = P2 ^ 5;
sbit _CL_P2_6          = P2 ^ 6;
sbit _CL_P2_7          = P2 ^ 7;

sbit _CL_P3_0          = P3 ^ 0;
sbit _CL_P3_1          = P3 ^ 1;
sbit _CL_P3_2          = P3 ^ 2;
sbit _CL_P3_3          = P3 ^ 3;
sbit _CL_P3_4          = P3 ^ 4;
sbit _CL_P3_5          = P3 ^ 5;
sbit _CL_P3_6          = P3 ^ 6;
sbit _CL_P3_7          = P3 ^ 7;

#if CL_WaitFor
/* The following procedure determines the wait for the 2 phase communications */
static void WaitFor(uint t, uint oneus_loop)
{
   uint data i,j;
   for (i=t;i!=0;i--)
   {
      for(j=oneus_loop;j!=0;j--);
   }
}
#endif

#if CL_SetPortByte
static void _CL_SetPortByte(uchar address, uchar value)
{
   switch (address)
   {
      case 0x80: P0 = value; break;
      case 0x90: P1 = value; break;
      case 0xa0: P2 = value; break;
      case 0xb0: P3 = value; break;
   }
}
#endif

#if CL_GetPortByte
static uchar _CL_GetPortByte(uchar address)
{
   switch (address)
   {
      case 0x80: return(P0);
      case 0x90: return(P1);
      case 0xa0: return(P2);
      case 0xb0: return(P3);
   }
}
#endif

#if CL_SetPortBit
static void _CL_SetPortBit(uchar address, bit value)
{
   switch (address)
   {
      case 0x80: _CL_P0_0 = value; break;
      case 0x81: _CL_P0_1 = value; break;
      case 0x82: _CL_P0_2 = value; break;
      case 0x83: _CL_P0_3 = value; break;
      case 0x84: _CL_P0_4 = value; break;
      case 0x85: _CL_P0_5 = value; break;
      case 0x86: _CL_P0_6 = value; break;
      case 0x87: _CL_P0_7 = value; break;
 
      case 0x90: _CL_P1_0 = value; break;
      case 0x91: _CL_P1_1 = value; break;
      case 0x92: _CL_P1_2 = value; break;
      case 0x93: _CL_P1_3 = value; break;
      case 0x94: _CL_P1_4 = value; break;
      case 0x95: _CL_P1_5 = value; break;
      case 0x96: _CL_P1_6 = value; break;
      case 0x97: _CL_P1_7 = value; break;

      case 0xa0: _CL_P2_0 = value; break;
      case 0xa1: _CL_P2_1 = value; break;
      case 0xa2: _CL_P2_2 = value; break;
      case 0xa3: _CL_P2_3 = value; break;
      case 0xa4: _CL_P2_4 = value; break;
      case 0xa5: _CL_P2_5 = value; break;
      case 0xa6: _CL_P2_6 = value; break;
      case 0xa7: _CL_P2_7 = value; break;

      case 0xb0: _CL_P3_0 = value; break;
      case 0xb1: _CL_P3_1 = value; break;
      case 0xb2: _CL_P3_2 = value; break;
      case 0xb3: _CL_P3_3 = value; break;
      case 0xb4: _CL_P3_4 = value; break;
      case 0xb5: _CL_P3_5 = value; break;
      case 0xb6: _CL_P3_6 = value; break;
      case 0xb7: _CL_P3_7 = value; break;
   }
}
#endif

#if CL_GetPortBit
static bit _CL_GetPortBit(uchar address)
{
   switch (address)
   {
      case 0x80: return(_CL_P0_0);
      case 0x81: return(_CL_P0_1);
      case 0x82: return(_CL_P0_2);
      case 0x83: return(_CL_P0_3);
      case 0x84: return(_CL_P0_4);
      case 0x85: return(_CL_P0_5);
      case 0x86: return(_CL_P0_6);
      case 0x87: return(_CL_P0_7);
 
      case 0x90: return(_CL_P1_0);
      case 0x91: return(_CL_P1_1);
      case 0x92: return(_CL_P1_2);
      case 0x93: return(_CL_P1_3);
      case 0x94: return(_CL_P1_4);
      case 0x95: return(_CL_P1_5);
      case 0x96: return(_CL_P1_6);
      case 0x97: return(_CL_P1_7);

      case 0xa0: return(_CL_P2_0);
      case 0xa1: return(_CL_P2_1);
      case 0xa2: return(_CL_P2_2);
      case 0xa3: return(_CL_P2_3);
      case 0xa4: return(_CL_P2_4);
      case 0xa5: return(_CL_P2_5);
      case 0xa6: return(_CL_P2_6);
      case 0xa7: return(_CL_P2_7);

      case 0xb0: return(_CL_P3_0);
      case 0xb1: return(_CL_P3_1);
      case 0xb2: return(_CL_P3_2);
      case 0xb3: return(_CL_P3_3);
      case 0xb4: return(_CL_P3_4);
      case 0xb5: return(_CL_P3_5);
      case 0xb6: return(_CL_P3_6);
      case 0xb7: return(_CL_P3_7);
   }
}
#endif

/* Default ports */
#define _CL_default_data8_p     CL_P3 	// data 
#define _CL_default_req_p       CL_P1_1 // data request 
#define _CL_default_ack_p       CL_P1_3 // data/address acknowledge 
#define _CL_default_data4_0_p   CL_P3_0
#define _CL_default_data4_1_p   CL_P3_1
#define _CL_default_data4_2_p   CL_P3_2
#define _CL_default_data4_3_p   CL_P3_3
#define _CL_default_data1_p     CL_P3_0
#define _CL_default_addr_req_p  CL_P1_2 // address request line
#define _CL_default_addr8_p     CL_P3   // address/data lines are common
#define _CL_default_addr4_0_p   CL_P3_0
#define _CL_default_addr4_1_p   CL_P3_1
#define _CL_default_addr4_2_p   CL_P3_2
#define _CL_default_addr4_3_p   CL_P3_3
#define _CL_default_addr1_p     CL_P3_0

#define _CL_default_wait_time   5
#define _CL_default_oneus_loop  35

// PC serial handshake lines
#define _CL_default_dsr_p  	CL_P1_1
#define _CL_default_dtr_p  	CL_P1_2
#define _CL_default_cts_p  	CL_P1_3
#define _CL_default_rts_p  	CL_P1_4

/**********************************************************************

   INITIALIZATIONS

**********************************************************************/

/****************************************************************************** 
    8-bit data bus, 4-phase handshake (D8_P4) (request and acknowledge lines) 
******************************************************************************/ 

#if (CL_SEND_M_D8P4==1)

void CL_InitDefaultSendMstD8P4(CL_SEND_M_D8_P4 *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_M_D8P4==2)

void CL_InitSendMstD8P4(CL_SEND_M_D8_P4 *chan, uchar data8_p_address,
                                   uchar req_p_address, uchar ack_p_address)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_S_D8P4==1)

void CL_InitDefaultRecSrvD8P4(CL_REC_S_D8_P4 *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_S_D8P4==2)

void CL_InitRecSrvD8P4(CL_REC_S_D8_P4 *chan, uchar data8_p_address,
                                   uchar req_p_address, uchar ack_p_address)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_M_D8P4==1)

void CL_InitDefaultRecMstD8P4(CL_REC_M_D8_P4 *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_M_D8P4==2)

void CL_InitRecMstD8P4(CL_REC_M_D8_P4 *chan, uchar data8_p_address,
                                   uchar req_p_address, uchar ack_p_address)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_S_D8P4==1)

void CL_InitDefaultSendSrvD8P4(CL_SEND_S_D8_P4 *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_SEND_S_D8P4==2)

void CL_InitSendSrvD8P4(CL_SEND_S_D8_P4 *chan, uchar data8_p_address,
                                   uchar req_p_address, uchar ack_p_address)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_M_D8P4_A==1)

void CL_InitDefaultSendMstD8P4A(CL_SEND_M_D8_P4_A *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr_p      = _CL_default_addr8_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_M_D8P4_A==2)

void CL_InitSendMstD8P4A(CL_SEND_M_D8_P4_A *chan, uchar data8_p_address,
                         uchar req_p_address, uchar ack_p_address,
                         uchar addr8_p_address, uchar addr_req_p_address)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr_p      = addr8_p_address;
   chan.addr_req_p  = addr_req_p_address;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_S_D8P4_A==1)

void CL_InitDefaultRecSrvD8P4A(CL_REC_S_D8_P4_A *chan, uchar addr)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr_p      = _CL_default_addr8_p;
   chan.addr        = addr;               // this receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortByte(chan.addr_p,    0xff); // addr lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_S_D8P4_A==2)

void CL_InitRecSrvD8P4A (CL_REC_S_D8_P4_A *chan, uchar data8_p_address,
        uchar req_p_address, uchar ack_p_address, uchar addr8_p_address, 
                              uchar addr_req_p_address, uchar addr)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr_p      = addr8_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortByte(chan.addr_p,    0xff); // addr lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_M_D8P4_A==1)

void CL_InitDefaultRecMstD8P4A(CL_REC_M_D8_P4_A *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr_p      = _CL_default_addr8_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_M_D8P4_A==2)

void CL_InitRecMstD8P4A(CL_REC_M_D8_P4_A *chan, uchar data8_p_address,
                         uchar req_p_address, uchar ack_p_address,
                         uchar addr8_p_address, uchar addr_req_p_address)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr_p      = addr8_p_address;
   chan.addr_req_p  = addr_req_p_address;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_S_D8P4_A==1)

void CL_InitDefaultSendSrvD8P4A(CL_SEND_S_D8_P4_A *chan, uchar addr)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr_p      = _CL_default_addr8_p;
   chan.addr        = addr;               // the receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortByte(chan.addr_p,    0xff); // addr lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_SEND_S_D8P4_A==2)

void CL_InitSendSrvD8P4A(CL_SEND_S_D8_P4_A *chan, uchar data8_p_address,
                         uchar req_p_address, uchar ack_p_address,
     uchar addr8_p_address, uchar addr_req_p_address, uchar addr)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr_p      = addr8_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;               // the receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortByte(chan.addr_p,    0xff); // addr lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

/****************************************************************************** 
    4-bit data bus, 4-phase handshake (D4_P4) (request and acknowledge lines) 
******************************************************************************/ 

#if (CL_SEND_M_D4P4==1)

void CL_InitDefaultSendMstD4P4(CL_SEND_M_D4_P4 *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_M_D4P4==2)

void CL_InitSendMstD4P4(CL_SEND_M_D4_P4 *chan, uchar data4_p_base_address,
                           uchar req_p_address, uchar ack_p_address)
{
   chan.data4_p      = data4_p_base_address;
   chan.req_p        = req_p_address;
   chan.ack_p        = ack_p_address;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_S_D4P4==1)

void CL_InitDefaultRecSrvD4P4(CL_REC_S_D4_P4 *chan)
{
   chan.data4_p        = _CL_default_data4_0_p;
   chan.req_p          = _CL_default_req_p;
   chan.ack_p          = _CL_default_ack_p;

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_S_D4P4==2)

void CL_InitRecSrvD4P4(CL_REC_S_D4_P4 *chan, uchar data4_p_base_address,
                           uchar req_p_address, uchar ack_p_address)
{
   chan.data4_p      = data4_p_base_address;
   chan.req_p        = req_p_address;
   chan.ack_p        = ack_p_address;

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_M_D4P4==1)

void CL_InitDefaultRecMstD4P4(CL_REC_M_D4_P4 *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_M_D4P4==2)

void CL_InitRecMstD4P4(CL_REC_M_D4_P4 *chan, uchar data4_p_base_address,
                           uchar req_p_address, uchar ack_p_address)
{
   chan.data4_p      = data4_p_base_address;
   chan.req_p        = req_p_address;
   chan.ack_p        = ack_p_address;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_S_D4P4==1)

void CL_InitDefaultSendSrvD4P4(CL_SEND_S_D4_P4 *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_SEND_S_D4P4==2)

void CL_InitSendSrvD4P4(CL_SEND_S_D4_P4 *chan, uchar data4_p_base_address,
                           uchar req_p_address, uchar ack_p_address)
{
   chan.data4_p      = data4_p_base_address;
   chan.req_p        = req_p_address;
   chan.ack_p        = ack_p_address;

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_SEND_M_D4P4_A==1)

void CL_InitDefaultSendMstD4P4A(CL_SEND_M_D4_P4_A *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.addr4_p     = _CL_default_addr4_0_p;

   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_M_D4P4_A==2)

void CL_InitSendMstD4P4A(CL_SEND_M_D4_P4_A *chan, uchar data4_p_base_address,
  uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                                            uchar addr_req_p_address)
{
   chan.data4_p     = data4_p_base_address;
   chan.addr4_p     = addr4_p_base_address;

   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr_req_p  = addr_req_p_address; 

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_S_D4P4_A==1)

void CL_InitDefaultRecSrvD4P4A(CL_REC_S_D4_P4_A *chan, uchar addr)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.addr4_p     = _CL_default_addr4_0_p;

   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr        = addr;               // this receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);    // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_S_D4P4_A==2)

void CL_InitRecSrvD4P4A(CL_REC_S_D4_P4_A *chan, uchar data4_p_base_address,
  uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                       uchar addr_req_p_address, uchar addr)
{
   chan.data4_p     = data4_p_base_address;
   chan.addr4_p     = addr4_p_base_address;

   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);    // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_M_D4P4_A==1)

void CL_InitDefaultRecMstD4P4A(CL_REC_M_D4_P4_A *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.addr4_p     = _CL_default_addr4_0_p;
   
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_M_D4P4_A==2)

void CL_InitRecMstD4P4A(CL_REC_M_D4_P4_A *chan, uchar data4_p_base_address,
  uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                                            uchar addr_req_p_address)
{
   chan.data4_p     = data4_p_base_address;
   chan.addr4_p     = addr4_p_base_address;

   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr_req_p  = addr_req_p_address; 
   
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_S_D4P4_A==1)

void CL_InitDefaultSendSrvD4P4A(CL_SEND_S_D4_P4_A *chan, uchar addr)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.addr4_p     = _CL_default_addr4_0_p;
   
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   
   chan.addr        = addr;               // the receiver's address
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);    // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_SEND_S_D4P4_A==2)

void CL_InitSendSrvD4P4A(CL_SEND_S_D4_P4_A *chan, uchar data4_p_base_address,
  uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                       uchar addr_req_p_address, uchar addr)
{
   chan.data4_p     = data4_p_base_address;
   chan.addr4_p     = addr4_p_base_address;

   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);    // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

/****************************************************************************** 
    1-bit data bus, 4-phase handshake (D1_P4) (request and acknowledge lines) 
******************************************************************************/ 

#if (CL_SEND_M_D1P4==1)

void CL_InitDefaultSendMstD1P4(CL_SEND_M_D1_P4 *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}
#endif

#if (CL_SEND_M_D1P4==2)

void CL_InitSendMstD1P4(CL_SEND_M_D1_P4 *chan, uchar data1_p_address,
                          uchar req_p_address, uchar ack_p_address)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_S_D1P4==1)

void CL_InitDefaultRecSrvD1P4(CL_REC_S_D1_P4 *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_S_D1P4==2)

void CL_InitRecSrvD1P4(CL_REC_S_D1_P4 *chan, uchar data1_p_address,
                              uchar req_p_address, uchar ack_p_address)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_M_D1P4==1)

void CL_InitDefaultRecMstD1P4(CL_REC_M_D1_P4 *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_M_D1P4==2)

void CL_InitRecMstD1P4(CL_REC_M_D1_P4 *chan, uchar data1_p_address,
                              uchar req_p_address, uchar ack_p_address)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_S_D1P4==1)

void CL_InitDefaultSendSrvD1P4(CL_SEND_S_D1_P4 *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_SEND_S_D1P4==2)

void CL_InitSendSrvD1P4(CL_SEND_S_D1_P4 *chan, uchar data1_p_address,
                              uchar req_p_address, uchar ack_p_address)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_SEND_M_D1P4_A==1)

void CL_InitDefaultSendMstD1P4A(CL_SEND_M_D1_P4_A *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;

   chan.addr4_p     = _CL_default_addr4_0_p;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_M_D1P4_A==2)

void CL_InitSendMstD1P4A(CL_SEND_M_D1_P4_A *chan, uchar data1_p_address,
                         uchar req_p_address, uchar ack_p_address,
                         uchar addr4_p_base_address, uchar addr_req_p_address)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr4_p     = addr4_p_base_address;
   chan.addr_req_p  = addr_req_p_address;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_S_D1P4_A==1)

void CL_InitDefaultRecSrvD1P4A(CL_REC_S_D1_P4_A *chan, uchar addr)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr4_p     = _CL_default_addr4_0_p;
   chan.addr        = addr;               // this receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);    // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_S_D1P4_A==2)

void CL_InitRecSrvD1P4A(CL_REC_S_D1_P4_A *chan, uchar data1_p_address,
      uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                      uchar addr_req_p_address, uchar addr)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr4_p     = addr4_p_base_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;               // this receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);      // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_REC_M_D1P4_A==1)

void CL_InitDefaultRecMstD1P4A(CL_REC_M_D1_P4_A *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr4_p     = _CL_default_addr4_0_p;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_REC_M_D1P4_A==2)

void CL_InitRecMstD1P4A(CL_REC_M_D1_P4_A *chan, uchar data1_p_address,
                         uchar req_p_address, uchar ack_p_address,
                         uchar addr4_p_base_address, uchar addr_req_p_address)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr4_p     = addr4_p_base_address;
   chan.addr_req_p  = addr_req_p_address;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
   _CL_SetPortBit (chan.ack_p, 1);        // read only
}

#endif

#if (CL_SEND_S_D1P4_A==1)

void CL_InitDefaultSendSrvD1P4A(CL_SEND_S_D1_P4_A *chan, uchar addr)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.ack_p       = _CL_default_ack_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr4_p     = _CL_default_addr4_0_p;
   chan.addr        = addr;               // the receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);     // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

#if (CL_SEND_S_D1P4_A==2)

void CL_InitSendSrvD1P4A(CL_SEND_S_D1_P4_A *chan, uchar data1_p_address,
      uchar req_p_address, uchar ack_p_address, uchar addr4_p_base_address,
                      uchar addr_req_p_address, uchar addr)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.ack_p       = ack_p_address;
   chan.addr4_p     = addr4_p_base_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;               // the receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);      // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
   _CL_SetPortBit (chan.ack_p,     1);    // active low, inactive state
}

#endif

/****************************************************************************** 
    8-bit data bus, 2-phase handshake (D8_P2) (req line only, fixed wait time) 
******************************************************************************/ 

#if (CL_SEND_M_D8P2==1)

void CL_InitDefaultSendMstD8P2(CL_SEND_M_D8_P2 *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_SEND_M_D8P2==2)

void CL_InitSendMstD8P2(CL_SEND_M_D8_P2 *chan, uchar data8_p_address,
       uchar req_p_address, uint wait_time, uint oneus_loop)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.wait_time   = wait_time;
   chan.oneus_loop  = oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_REC_S_D8P2==1)

void CL_InitDefaultRecSrvD8P2(CL_REC_S_D8_P2 *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_REC_S_D8P2==2)

void CL_InitRecSrvD8P2(CL_REC_S_D8_P2 *chan, uchar data8_p_address,
                                             uchar req_p_address)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_REC_M_D8P2==1)

void CL_InitDefaultRecMstD8P2(CL_REC_M_D8_P2 *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_REC_M_D8P2==2)

void CL_InitRecMstD8P2(CL_REC_M_D8_P2 *chan, uchar data8_p_address,
       uchar req_p_address, uint wait_time, uint oneus_loop)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.wait_time   = wait_time;
   chan.oneus_loop  = oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_SEND_S_D8P2==1)

void CL_InitDefaultSendSrvD8P2(CL_SEND_S_D8_P2 *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_SEND_S_D8P2==2)

void CL_InitSendSrvD8P2(CL_SEND_S_D8_P2 *chan, uchar data8_p_address,
                                             uchar req_p_address)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_SEND_M_D8P2_A==1)

void CL_InitDefaultSendMstD8P2A(CL_SEND_M_D8_P2_A *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr_p      = _CL_default_addr8_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_SEND_M_D8P2_A==2)

void CL_InitSendMstD8P2A(CL_SEND_M_D8_P2_A *chan, uchar data8_p_address,
   uchar req_p_address, uchar addr8_p_address, uchar addr_req_p_address,
                                       uint wait_time, uint oneus_loop)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.addr_p      = addr8_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.wait_time   = wait_time;
   chan.oneus_loop  = oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_REC_S_D8P2_A==1)

void CL_InitDefaultRecSrvD8P2A(CL_REC_S_D8_P2_A *chan, uchar addr)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr_p      = _CL_default_addr8_p;
   chan.addr        = addr;               // this receiver's address
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortByte(chan.addr_p,    0xff); // addr lines are read only
}

#endif

#if (CL_REC_S_D8P2_A==2)

void CL_InitRecSrvD8P2A(CL_REC_S_D8_P2_A *chan, uchar data8_p_address,
  uchar req_p_address, uchar addr8_p_address, uchar addr_req_p_address,
                                              uchar addr)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.addr_p      = addr8_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;               // this receiver's address
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortByte(chan.addr_p,    0xff); // addr lines are read only
}

#endif

#if (CL_REC_M_D8P2_A==1)

void CL_InitDefaultRecMstD8P2A(CL_REC_M_D8_P2_A *chan)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr_p      = _CL_default_addr8_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_REC_M_D8P2_A==2)

void CL_InitRecMstD8P2A(CL_REC_M_D8_P2_A *chan, uchar data8_p_address,
   uchar req_p_address, uchar addr8_p_address, uchar addr_req_p_address,
                                       uint wait_time, uint oneus_loop)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.addr_p      = addr8_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.wait_time   = wait_time;
   chan.oneus_loop  = oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_SEND_S_D8P2_A==1)

void CL_InitDefaultSendSrvD8P2A(CL_SEND_S_D8_P2_A *chan, uchar addr)
{
   chan.data_p      = _CL_default_data8_p;
   chan.req_p       = _CL_default_req_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr_p      = _CL_default_addr8_p;
   chan.addr        = addr;               // the receiver's address
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortByte(chan.addr_p,    0xff); // addr lines are read only
}

#endif

#if (CL_SEND_S_D8P2_A==2)

void CL_InitSendSrvD8P2A(CL_SEND_S_D8_P2_A *chan, uchar data8_p_address,
  uchar req_p_address, uchar addr8_p_address, uchar addr_req_p_address,
                                              uchar addr)
{
   chan.data_p      = data8_p_address;
   chan.req_p       = req_p_address;
   chan.addr_p      = addr8_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;               // the receiver's address
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortByte(chan.addr_p,    0xff); // addr lines are read only
}

#endif

/****************************************************************************** 
    4-bit data bus, 2-phase handshake (D4_P2) (req line only, fixed wait time) 
******************************************************************************/ 

#if (CL_SEND_M_D4P2==1)

void CL_InitDefaultSendMstD4P2(CL_SEND_M_D4_P2 *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_SEND_M_D4P2==2)

void CL_InitSendMstD4P2(CL_SEND_M_D4_P2 *chan, uchar data4_p_base_address,
         uchar req_p_address, uint wait_time, uint oneus_loop)
{
   chan.data4_p        = data4_p_base_address;
   chan.req_p          = req_p_address;
   chan.wait_time      = wait_time;
   chan.oneus_loop     = oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_REC_S_D4P2==1)

void CL_InitDefaultRecSrvD4P2(CL_REC_S_D4_P2 *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_REC_S_D4P2==2)

void CL_InitRecSrvD4P2(CL_REC_S_D4_P2 *chan, uchar data4_p_base_address,
                                                    uchar req_p_address)
{
   chan.data4_p        = data4_p_base_address;
   chan.req_p          = req_p_address;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_REC_M_D4P2==1)

void CL_InitDefaultRecMstD4P2(CL_REC_M_D4_P2 *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   chan.req_p       = _CL_default_req_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_REC_M_D4P2==2)

void CL_InitRecMstD4P2(CL_REC_M_D4_P2 *chan, uchar data4_p_base_address,
         uchar req_p_address, uint wait_time, uint oneus_loop)
{
   chan.data4_p        = data4_p_base_address;
   chan.req_p          = req_p_address;
   chan.wait_time      = wait_time;
   chan.oneus_loop     = oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_SEND_S_D4P2==1)

void CL_InitDefaultSendSrvD4P2(CL_SEND_S_D4_P2 *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_REC_M_D4P2==2)

void CL_InitSendSrvD4P2(CL_SEND_S_D4_P2 *chan, uchar data4_p_base_address,
                                              uchar req_p_address)
{
   chan.data4_p        = data4_p_base_address;
   chan.req_p          = req_p_address;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_SEND_M_D4P2_A==1)

void CL_InitDefaultSendMstD4P2A(CL_SEND_M_D4_P2_A *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr4_p     = _CL_default_addr4_0_p;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_SEND_M_D4P2_A==2)

void CL_InitSendMstD4P2A(CL_SEND_M_D4_P2_A *chan, uchar data4_p_base_address,
  uchar req_p_address, uchar addr4_p_base_address, uchar addr_req_p_address,
                                         uint wait_time, uint oneus_loop)
{
   chan.data4_p        = data4_p_base_address;
   chan.req_p          = req_p_address;
   chan.addr4_p        = addr4_p_base_address;
   chan.addr_req_p     = addr_req_p_address;
   chan.wait_time      = wait_time;
   chan.oneus_loop     = oneus_loop;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_REC_S_D4P2_A==1)

void CL_InitDefaultRecSrvD4P2A(CL_REC_S_D4_P2_A *chan, uchar addr)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr4_p     = _CL_default_addr4_0_p;
   chan.addr        = addr;               // this receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);      // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
}

#endif

#if (CL_REC_S_D4P2_A==2)

void CL_InitRecSrvD4P2A(CL_REC_S_D4_P2_A *chan, uchar data4_p_base_address,
   uchar req_p_address, uchar addr4_p_base_address, uchar addr_req_p_address,
                                                          uchar addr)
{
   chan.data4_p     = data4_p_base_address;
   chan.req_p       = req_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr4_p     = addr4_p_base_address;
   chan.addr        = addr;               // this receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);      // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
}

#endif

#if (CL_REC_M_D4P2_A==1)

void CL_InitDefaultRecMstD4P2A(CL_REC_M_D4_P2_A *chan)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr4_p     = _CL_default_addr4_0_p;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_REC_M_D4P2_A==2)

void CL_InitRecMstD4P2A(CL_REC_M_D4_P2_A *chan, uchar data4_p_base_address,
  uchar req_p_address, uchar addr4_p_base_address, uchar addr_req_p_address,
                                         uint wait_time, uint oneus_loop)
{
   chan.data4_p        = data4_p_base_address;
   chan.req_p          = req_p_address;
   chan.addr4_p        = addr4_p_base_address;
   chan.addr_req_p     = addr_req_p_address;
   chan.wait_time      = wait_time;
   chan.oneus_loop     = oneus_loop;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_SEND_S_D4P2_A==1)

void CL_InitDefaultSendSrvD4P2A(CL_SEND_S_D4_P2_A *chan, uchar addr)
{
   chan.data4_p     = _CL_default_data4_0_p;
   chan.req_p       = _CL_default_req_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr4_p     = _CL_default_addr4_0_p;
   chan.addr        = addr;               // the receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);    // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
}

#endif

#if (CL_SEND_S_D4P2_A==2)

void CL_InitSendSrvD4P2A(CL_SEND_S_D4_P2_A *chan, uchar data4_p_base_address,
   uchar req_p_address, uchar addr4_p_base_address,uchar addr_req_p_address,
                                                          uchar addr)
{
   chan.data4_p     = data4_p_base_address;
   chan.req_p       = req_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr4_p     = addr4_p_base_address;
   chan.addr        = addr;               // the receiver's address

   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr4_p, 1);      // addr lines are read only
   _CL_SetPortBit (chan.addr4_p+0x01, 1);
   _CL_SetPortBit (chan.addr4_p+0x02, 1);
   _CL_SetPortBit (chan.addr4_p+0x03, 1);
}

#endif

/****************************************************************************** 
    1-bit data bus, 2-phase handshake (D1_P2) (req line only, fixed wait time) 
******************************************************************************/ 

#if (CL_SEND_M_D1P2==1)

void CL_InitDefaultSendMstD1P2(CL_SEND_M_D1_P2 *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_SEND_M_D1P2==2)

void CL_InitSendMstD1P2(CL_SEND_M_D1_P2 *chan, uchar data1_p_address,
               uchar req_p_address, uint wait_time, uint oneus_loop)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.wait_time   = wait_time;
   chan.oneus_loop  = oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_REC_S_D1P2==1)

void CL_InitDefaultRecSrvD1P2(CL_REC_S_D1_P2 *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_REC_S_D1P2==2)

void CL_InitRecSrvD1P2(CL_REC_S_D1_P2 *chan, uchar data1_p_address,
                                             uchar req_p_address)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_REC_M_D1P2==1)

void CL_InitDefaultRecMstD1P2(CL_REC_M_D1_P2 *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_REC_M_D1P2==2)

void CL_InitRecMstD1P2(CL_REC_M_D1_P2 *chan, uchar data1_p_address,
               uchar req_p_address, uint wait_time, uint oneus_loop)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.wait_time   = wait_time;
   chan.oneus_loop  = oneus_loop;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
}

#endif

#if (CL_SEND_S_D1P2==1)

void CL_InitDefaultSendSrvD1P2(CL_SEND_S_D1_P2 *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_SEND_S_D1P2==2)

void CL_InitSendSrvD1P2(CL_SEND_S_D1_P2 *chan, uchar data1_p_address,
                                             uchar req_p_address)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
}

#endif

#if (CL_SEND_M_D1P2_A==1)

void CL_InitDefaultSendMstD1P2A(CL_SEND_M_D1_P2_A *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr1_p     = _CL_default_addr1_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_SEND_M_D1P2_A==2)

void CL_InitSendMstD1P2A(CL_SEND_M_D1_P2_A *chan, uchar data1_p_address,
        uchar req_p_address, uchar addr1_p_address, uchar addr_req_p_address,
        uint wait_time, uint oneus_loop)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.addr1_p     = addr1_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.wait_time   = wait_time;
   chan.oneus_loop  = oneus_loop;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_REC_S_D1P2_A==1)

void CL_InitDefaultRecSrvD1P2A(CL_REC_S_D1_P2_A *chan, uchar addr)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr1_p     = _CL_default_addr1_p;
   chan.addr        = addr;               // this receiver's address
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr1_p,   1);    // addr line is read only
}

#endif

#if (CL_REC_S_D1P2_A==2)

void CL_InitRecSrvD1P2A(CL_REC_S_D1_P2_A *chan, uchar data1_p_address,
        uchar req_p_address, uchar addr1_p_address, uchar addr_req_p_address,
                                                    uchar addr)    
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.addr1_p     = addr1_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr1_p,   1);    // addr line is read only
}

#endif

#if (CL_REC_M_D1P2_A==1)

void CL_InitDefaultRecMstD1P2A(CL_REC_M_D1_P2_A *chan)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.wait_time   = _CL_default_wait_time;
   chan.oneus_loop  = _CL_default_oneus_loop;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr1_p     = _CL_default_addr1_p;
   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_REC_M_D1P2_A==2)

void CL_InitRecMstD1P2A(CL_REC_M_D1_P2_A *chan, uchar data1_p_address,
        uchar req_p_address, uchar addr1_p_address, uchar addr_req_p_address,
        uint wait_time, uint oneus_loop)
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.addr1_p     = addr1_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.wait_time   = wait_time;
   chan.oneus_loop  = oneus_loop;

   _CL_SetPortBit (chan.req_p, 1);        // active low logic
   _CL_SetPortBit (chan.addr_req_p, 1); 
}

#endif

#if (CL_SEND_S_D1P2_A==1)

void CL_InitDefaultSendSrvD1P2A(CL_SEND_S_D1_P2_A *chan, uchar addr)
{
   chan.data1_p     = _CL_default_data1_p;
   chan.req_p       = _CL_default_req_p;
   chan.addr_req_p  = _CL_default_addr_req_p;
   chan.addr1_p     = _CL_default_addr1_p;
   chan.addr        = addr;               // this receiver's address
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr1_p,   1);    // addr line is read only
}

#endif

#if (CL_SEND_S_D1P2_A==2)

void CL_InitSendSrvD1P2A(CL_SEND_S_D1_P2_A *chan, uchar data1_p_address,
        uchar req_p_address, uchar addr1_p_address, uchar addr_req_p_address,
                                                    uchar addr)    
{
   chan.data1_p     = data1_p_address;
   chan.req_p       = req_p_address;
   chan.addr1_p     = addr1_p_address;
   chan.addr_req_p  = addr_req_p_address;
   chan.addr        = addr;
   _CL_SetPortBit (chan.req_p,     1);    // req lines are read only
   _CL_SetPortBit (chan.addr_req_p,1); 
   _CL_SetPortBit (chan.addr1_p,   1);    // addr line is read only
}

#endif


/**********************************************************************

   TRANSFERS

**********************************************************************/

/****************************************************************************** 
    8-bit data bus, 4-phase handshake (D8_P4) (request and acknowledge lines) 
******************************************************************************/ 

#if CL_SEND_M_D8P4

void CL_SendMstD8P4(CL_SEND_M_D8_P4 chan, uchar char_to_send)
{
   _CL_SetPortByte(chan.data_p, char_to_send);  // Put data on port
   _CL_SetPortBit (chan.req_p,  0);             // Tell receiver data is ready
   while (_CL_GetPortBit(chan.ack_p));          // Wait until data received
   _CL_SetPortBit (chan.req_p,  1);             // Finish 4-phase handshake
   while (!_CL_GetPortBit(chan.ack_p));

   _CL_SetPortByte(chan.data_p,0xff);           // Release the bus
   return;
}

#endif

#if CL_REC_S_D8P4

uchar CL_RecSrvD8P4(CL_REC_S_D8_P4 chan)
{
   uchar data char_received=0x00;

   while (_CL_GetPortBit(chan.req_p));          // Wait until data is ready
   char_received = _CL_GetPortByte(chan.data_p);// Get the data
   _CL_SetPortBit (chan.ack_p, 0);              // Tell sender data received
   while (!_CL_GetPortBit(chan.req_p));         // Finish the 4-phase handshake
   _CL_SetPortBit (chan.ack_p, 1);

   return char_received;
}

#endif

#if CL_REC_M_D8P4

uchar CL_RecMstD8P4(CL_REC_M_D8_P4 chan)
{
   uchar data char_received=0x00;

   _CL_SetPortBit (chan.req_p,  0);             // Tell sender ready to receive 
   while (_CL_GetPortBit(chan.ack_p));          // Wait until data is ready
   char_received = _CL_GetPortByte(chan.data_p);// Get the data
   _CL_SetPortBit (chan.req_p,  1);             // Tell sender received data
   while (!_CL_GetPortBit(chan.ack_p));         // Finish 4-phase handshake

   return char_received;
}

#endif

#if CL_SEND_S_D8P4

void CL_SendSrvD8P4(CL_SEND_S_D8_P4 chan, uchar char_to_send)
{
   while (_CL_GetPortBit(chan.req_p));          // Wait for master request data
   _CL_SetPortByte(chan.data_p,char_to_send);   // Put the data on the port
   _CL_SetPortBit (chan.ack_p, 0);              // Tell master data is ready
   while (!_CL_GetPortBit(chan.req_p));         // Wait until master receives 
   _CL_SetPortBit (chan.ack_p, 1);              // Finish the 4-phase handshake

   _CL_SetPortByte(chan.data_p,0xff);           // Set data port to high impedance
   return;
}

#endif

#if CL_SEND_M_D8P4_A

void CL_SendMstD8P4A(CL_SEND_M_D8_P4_A chan, uchar addr, uchar char_to_send)
{
   _CL_SetPortByte(chan.addr_p, addr);          // Put address on port
   _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
   while (_CL_GetPortBit(chan.ack_p));          // wait until address received 
   _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 4-phase handshake
   while (!_CL_GetPortBit(chan.ack_p));   

   _CL_SetPortByte(chan.data_p, char_to_send);  // Put data on port
   _CL_SetPortBit (chan.req_p,  0);             // Tell receiver data is ready
   while (_CL_GetPortBit(chan.ack_p));          // Wait until data received
   _CL_SetPortBit (chan.req_p,  1);             // Finish 4-phase handshake
   while (!_CL_GetPortBit(chan.ack_p));

   _CL_SetPortByte(chan.data_p,0xff);           // Release the bus
   return;
}

#endif

#if CL_REC_S_D8P4_A

uchar CL_RecSrvD8P4A(CL_REC_S_D8_P4_A chan)
{
   uchar data char_received=0x00;
   uchar data addr_received=0x00;

   while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready
   addr_received = _CL_GetPortByte(chan.addr_p);// Get the address

   while(addr_received != chan.addr)
   {
      _CL_SetPortBit (chan.ack_p, 1);           // set ack to high impedance
      addr_received = 0x00;              
      while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready
      addr_received = _CL_GetPortByte(chan.addr_p);// Get the address
   }   

   _CL_SetPortBit (chan.ack_p, 0);              // Tell sender address received
   while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 4-phase handshake
   _CL_SetPortBit (chan.ack_p, 1);

   while (_CL_GetPortBit(chan.req_p));          // Wait until data is ready
   char_received = _CL_GetPortByte(chan.data_p);// Get the data
   _CL_SetPortBit (chan.ack_p, 0);              // Tell sender data received
   while (!_CL_GetPortBit(chan.req_p));         // Finish the 4-phase handshake
   _CL_SetPortBit (chan.ack_p, 1);

   return char_received;
}

#endif

#if CL_REC_M_D8P4_A

uchar CL_RecMstD8P4A(CL_REC_M_D8_P4_A chan, uchar addr)
{
   uchar data char_received=0x00;

   _CL_SetPortByte(chan.addr_p, addr);          // Put address on port
   _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
   while (_CL_GetPortBit(chan.ack_p));          // wait until address received 
   _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 4-phase handshake
   while (!_CL_GetPortBit(chan.ack_p));   

   _CL_SetPortByte(chan.data_p,0xff);           // Write all 1's before reading 

   _CL_SetPortBit (chan.req_p,  0);             // Tell sender ready to receive 
   while (_CL_GetPortBit(chan.ack_p));          // Wait until data is ready
   char_received = _CL_GetPortByte(chan.data_p);// Get the data
   _CL_SetPortBit (chan.req_p,  1);             // Tell sender received data
   while (!_CL_GetPortBit(chan.ack_p));         // Finish 4-phase handshake

   return char_received;
}

#endif

#if CL_SEND_S_D8P4_A

void CL_SendSrvD8P4A(CL_SEND_S_D8_P4_A chan, uchar char_to_send)
{
   uchar data addr_received=0x00;

   while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready
   addr_received = _CL_GetPortByte(chan.addr_p);// Get the address

   while(addr_received != chan.addr)
   {
      _CL_SetPortBit (chan.ack_p, 1);           // set ack to high impedance
      addr_received = 0x00;              
      while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready
      addr_received = _CL_GetPortByte(chan.addr_p);// Get the address
   }   

   _CL_SetPortBit (chan.ack_p, 0);              // Tell sender address received
   while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 4-phase handshake
   _CL_SetPortBit (chan.ack_p, 1);

   while (_CL_GetPortBit(chan.req_p));          // Wait for master request data
   _CL_SetPortByte(chan.data_p,char_to_send);   // Put the data on the port
   _CL_SetPortBit (chan.ack_p, 0);              // Tell master data is ready
   while (!_CL_GetPortBit(chan.req_p));         // Wait until master receives 
   _CL_SetPortBit (chan.ack_p, 1);              // Finish the 4-phase handshake

   _CL_SetPortByte(chan.data_p,0xff);           // Set data port to high impedance
   return;
}

#endif

/****************************************************************************** 
    4-bit data bus, 4-phase handshake (D4_P4) (request and acknowledge lines) 
******************************************************************************/ 

#if CL_SEND_M_D4P4

void CL_SendMstD4P4(CL_SEND_M_D4_P4 chan, uchar char_to_send)
{
   ushort data i;

   for(i=0;i<2;++i)
   {
      // Put the data on the port
      _CL_SetPortBit(chan.data4_p, (bit) (char_to_send >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.data4_p + 0x01, (bit) (char_to_send >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.data4_p + 0x02, (bit) (char_to_send >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.data4_p + 0x03, (bit) (char_to_send >> (4*i + 3) & 0x01));  

      _CL_SetPortBit (chan.req_p,  0);          // Tell receiver data is ready
      while (_CL_GetPortBit(chan.ack_p));       // Wait until data received
      _CL_SetPortBit (chan.req_p,  1);          // Finish 4-phase handshake
      while (!_CL_GetPortBit(chan.ack_p));
   }

   //Release the bus
   CL_SetPortBit(chan.data4_p, 1);
   CL_SetPortBit(chan.data4_p + 0x01, 1);
   CL_SetPortBit(chan.data4_p + 0x02, 1);
   CL_SetPortBit(chan.data4_p + 0x03, 1);

   return;
}

#endif

#if CL_REC_S_D4P4

uchar CL_RecSrvD4P4(CL_REC_S_D4_P4 chan)
{
   ushort data i;
   uchar data char_received=0x00;

   for (i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.req_p));       // Wait until data is ready

      char_received |= (char)_CL_GetPortBit(chan.data4_p) << (4*i);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x01) << (4*i + 1);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x02) << (4*i + 2);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x03) << (4*i + 3);

      _CL_SetPortBit (chan.ack_p, 0);           // Tell sender data received
      while (!_CL_GetPortBit(chan.req_p));      // Finish the 4-phase handshake
      _CL_SetPortBit (chan.ack_p, 1);
   }

   return char_received;
}

#endif

#if CL_REC_M_D4P4

uchar CL_RecMstD4P4(CL_REC_M_D4_P4 chan)
{
   ushort data i;
   uchar data char_received=0x00;

   for(i=0;i<2;++i)
   {
      _CL_SetPortBit (chan.req_p,  0);          // Tell sender ready to receive 
      while (_CL_GetPortBit(chan.ack_p));       // Wait until data is ready

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data4_p) << (4*i);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x01) << (4*i + 1);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x02) << (4*i + 2);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x03) << (4*i + 3);

      _CL_SetPortBit (chan.req_p,  1);          // Tell sender received data
      while (!_CL_GetPortBit(chan.ack_p));      // Finish 4-phase handshake
   }

   return char_received;
}

#endif

#if CL_SEND_S_D4P4

void CL_SendSrvD4P4(CL_SEND_S_D4_P4 chan, uchar char_to_send)
{
   ushort data i;

   for(i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.req_p));       // Wait for master request data

      //Put the data on the port
      _CL_SetPortBit(chan.data4_p, (bit) (char_to_send >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x01, (bit) (char_to_send >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x02, (bit) (char_to_send >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x03, (bit) (char_to_send >> (4*i + 3) & 0x01));  

      _CL_SetPortBit (chan.ack_p, 0);           // Tell master data is ready
      while (!_CL_GetPortBit(chan.req_p));      // Wait until master receives 
      _CL_SetPortBit (chan.ack_p, 1);           // Finish the 4-phase handshake
   }

   //Set all data port bits to high impedance
   _CL_SetPortBit(chan.data4_p,1);
   _CL_SetPortBit(chan.data4_p+0x01,1); 
   _CL_SetPortBit(chan.data4_p+0x02,1); 
   _CL_SetPortBit(chan.data4_p+0x03,1); 

   return;
}

#endif

#if CL_SEND_M_D4P4_A

void CL_SendMstD4P4A(CL_SEND_M_D4_P4_A chan, uchar addr, uchar char_to_send)
{
   ushort data i;

   //Put the 4 bit address on the port
   _CL_SetPortBit(chan.addr4_p, (bit)  (addr & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x01, (bit)  (addr >> 1 & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x02, (bit)  (addr >> 2 & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x03, (bit)  (addr >> 4 & 0x01));  

   _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
   while (_CL_GetPortBit(chan.ack_p));          // wait until address received 
   _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 4-phase handshake
   while (!_CL_GetPortBit(chan.ack_p));   

   for(i=0;i<2;++i)
   {
      //Put the data on the port
      _CL_SetPortBit(chan.data4_p, (bit) (char_to_send >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x01, (bit) (char_to_send >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x02, (bit) (char_to_send >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x03, (bit) (char_to_send >> (4*i + 3) & 0x01));  
   
      _CL_SetPortBit (chan.req_p,  0);             // Tell receiver data is ready
      while (_CL_GetPortBit(chan.ack_p));          // Wait until data received
      _CL_SetPortBit (chan.req_p,  1);             // Finish 4-phase handshake
      while (!_CL_GetPortBit(chan.ack_p));
   }

   //Release the bus
   _CL_SetPortBit(chan.data4_p, 1);
   _CL_SetPortBit(chan.data4_p + 0x01, 1);
   _CL_SetPortBit(chan.data4_p + 0x02, 1);
   _CL_SetPortBit(chan.data4_p + 0x03, 1);

   return;
}

#endif

#if CL_REC_S_D4P4_A

uchar CL_RecSrvD4P4A(CL_REC_S_D4_P4_A chan)
{
   ushort data i;
   uchar data char_received=0x00;
   uchar data addr_received=0x00;

   while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready

   //get the address
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p);
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << 1;
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << 2;
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << 3;

   while(addr_received != chan.addr)
   {
      _CL_SetPortBit (chan.ack_p, 1);           // set ack to high impedance
      addr_received = 0x00;              
      while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready
      //get the address
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << 1;
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << 2;
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << 3;
   }   

   _CL_SetPortBit (chan.ack_p, 0);              // Tell sender address received
   while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 4-phase handshake
   _CL_SetPortBit (chan.ack_p, 1);

   for(i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.req_p));          // Wait until data is ready

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data4_p) << (4*i);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x01) << (4*i + 1);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x02) << (4*i + 2);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x03) << (4*i + 3);

      _CL_SetPortBit (chan.ack_p, 0);              // Tell sender data received
      while (!_CL_GetPortBit(chan.req_p));         // Finish the 4-phase handshake
      _CL_SetPortBit (chan.ack_p, 1);
   }

   return char_received;
}

#endif

#if CL_REC_M_D4P4_A

uchar CL_RecMstD4P4A(CL_REC_M_D4_P4_A chan, uchar addr)
{
   ushort data i;
   uchar data char_received=0x00;

   //Put the 4 bit address on the port
   _CL_SetPortBit(chan.addr4_p, (bit)  (addr & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x01, (bit)  (addr >> 1 & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x02, (bit)  (addr >> 2 & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x03, (bit)  (addr >> 4 & 0x01));  
   
   _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
   while (_CL_GetPortBit(chan.ack_p));          // wait until address received 
   _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 4-phase handshake
   while (!_CL_GetPortBit(chan.ack_p));   

   //Set all data port bits to 1 to read
   _CL_SetPortBit(chan.data4_p,1);
   _CL_SetPortBit(chan.data4_p+0x01,1); 
   _CL_SetPortBit(chan.data4_p+0x02,1); 
   _CL_SetPortBit(chan.data4_p+0x03,1); 

   for(i=0;i<2;++i)
   {
      _CL_SetPortBit (chan.req_p,  0);             // Tell sender ready to receive 
      while (_CL_GetPortBit(chan.ack_p));          // Wait until data is ready

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data4_p) << (4*i);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x01) << (4*i + 1);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x02) << (4*i + 2);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x03) << (4*i + 3);

      _CL_SetPortBit (chan.req_p,  1);             // Tell sender received data
      while (!_CL_GetPortBit(chan.ack_p));         // Finish 4-phase handshake
   }

   return char_received;
}

#endif

#if CL_SEND_S_D4P4_A

void CL_SendSrvD4P4A(CL_SEND_S_D4_P4_A chan, uchar char_to_send)
{
   ushort data i;
   uchar data addr_received=0x00;

   while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready

   //get the address
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p);
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << 1;
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << 2;
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << 3;

   while(addr_received != chan.addr)
   {
      _CL_SetPortBit (chan.ack_p, 1);           // set ack to high impedance
      addr_received = 0x00;              
      while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready
      //get the address
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << 1;
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << 2;
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << 3;
   }   

   _CL_SetPortBit (chan.ack_p, 0);              // Tell sender address received
   while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 4-phase handshake
   _CL_SetPortBit (chan.ack_p, 1);

   for(i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.req_p));          // Wait for master request data

      //Put the data on the port
      _CL_SetPortBit(chan.data4_p, (bit) (char_to_send >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x01, (bit) (char_to_send >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x02, (bit) (char_to_send >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x03, (bit) (char_to_send >> (4*i + 3) & 0x01));  

      _CL_SetPortBit (chan.ack_p, 0);              // Tell master data is ready
      while (!_CL_GetPortBit(chan.req_p));         // Wait until master receives 
      _CL_SetPortBit (chan.ack_p, 1);              // Finish the 4-phase handshake
   }

   //Set all data port bits to high impedance
   _CL_SetPortBit(chan.data4_p,1);
   _CL_SetPortBit(chan.data4_p+0x01,1); 
   _CL_SetPortBit(chan.data4_p+0x02,1); 
   _CL_SetPortBit(chan.data4_p+0x03,1); 

   return;
}

#endif

/****************************************************************************** 
    1-bit data bus, 4-phase handshake (D1_P4) (request and acknowledge lines) 
******************************************************************************/ 

#if CL_SEND_M_D1P4

void CL_SendMstD1P4(CL_SEND_M_D1_P4 chan, uchar char_to_send)
{
   ushort data i;

   for(i=0;i<8;++i)
   {
      // Put the data on the port
      _CL_SetPortBit(chan.data1_p, (bit) ((char_to_send >> i) & 0x01));  

      _CL_SetPortBit (chan.req_p,  0);          // Tell receiver data is ready
      while (_CL_GetPortBit(chan.ack_p));       // Wait until data received
      _CL_SetPortBit (chan.req_p,  1);          // Finish 4-phase handshake
      while (!_CL_GetPortBit(chan.ack_p));
   }

   //Set data port to high impedance
   _CL_SetPortBit(chan.data1_p,1);

   return;
}

#endif

#if CL_REC_S_D1P4

uchar CL_RecSrvD1P4(CL_REC_S_D1_P4 chan)
{
   ushort data i;
   uchar data char_received=0x00;

   for (i=0;i<8;++i)
   {
      while (_CL_GetPortBit(chan.req_p));       // Wait until data is ready

      char_received |= (char)_CL_GetPortBit(chan.data1_p) << i;

      _CL_SetPortBit (chan.ack_p, 0);           // Tell sender data received
      while (!_CL_GetPortBit(chan.req_p));      // Finish the 4-phase handshake
      _CL_SetPortBit (chan.ack_p, 1);
   }

   return char_received;
}

#endif

#if CL_REC_M_D1P4

uchar CL_RecMstD1P4(CL_REC_M_D1_P4 chan)
{
   ushort data i;
   uchar data char_received=0x00;

   for(i=0;i<8;++i)
   {
      _CL_SetPortBit (chan.req_p,  0);          // Tell sender ready to receive 
      while (_CL_GetPortBit(chan.ack_p));       // Wait until data is ready

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data1_p) << i;

      _CL_SetPortBit (chan.req_p,  1);          // Tell sender received data
      while (!_CL_GetPortBit(chan.ack_p));      // Finish 4-phase handshake
   }

   return char_received;
}

#endif

#if CL_SEND_S_D1P4

void CL_SendSrvD1P4(CL_SEND_S_D1_P4 chan, uchar char_to_send)
{
   ushort data i;

   for(i=0;i<8;++i)
   {
      while (_CL_GetPortBit(chan.req_p));       // Wait for master request data

      //Put the data on the port
      _CL_SetPortBit(chan.data1_p, (bit) ((char_to_send >> i) & 0x01));  

      _CL_SetPortBit (chan.ack_p, 0);           // Tell master data is ready
      while (!_CL_GetPortBit(chan.req_p));      // Wait until master receives 
      _CL_SetPortBit (chan.ack_p, 1);           // Finish the 4-phase handshake
   }

   //Set all data port to high impedance
   _CL_SetPortBit(chan.data1_p,1);

   return;
}

#endif

#if CL_SEND_M_D1P4_A

void CL_SendMstD1P4A(CL_SEND_M_D1_P4_A chan, uchar addr, uchar char_to_send)
{
   ushort data i;

   //Put the 4 bit address on the port
   _CL_SetPortBit(chan.addr4_p, (bit)  (addr & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x01, (bit)  (addr >> 1 & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x02, (bit)  (addr >> 2 & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x03, (bit)  (addr >> 4 & 0x01));  

   _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
   while (_CL_GetPortBit(chan.ack_p));          // wait until address received 
   _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 4-phase handshake
   while (!_CL_GetPortBit(chan.ack_p));   

   for(i=0;i<8;++i)
   {
      //Put the data on the port
      _CL_SetPortBit(chan.data1_p, (bit) ((char_to_send >> i) & 0x01));  
   
      _CL_SetPortBit (chan.req_p,  0);             // Tell receiver data is ready
      while (_CL_GetPortBit(chan.ack_p));          // Wait until data received
      _CL_SetPortBit (chan.req_p,  1);             // Finish 4-phase handshake
      while (!_CL_GetPortBit(chan.ack_p));
   }

   //Set data port to high impedance
   _CL_SetPortBit(chan.data1_p,1);

   return;
}

#endif

#if CL_REC_S_D1P4_A

uchar CL_RecSrvD1P4A(CL_REC_S_D1_P4_A chan)
{
   ushort data i;
   uchar data char_received=0x00;
   uchar data addr_received=0x00;

   while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready

   //get the address
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p);
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << 1;
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << 2;
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << 3;

   while(addr_received != chan.addr)
   {
      _CL_SetPortBit (chan.ack_p, 1);           // set ack to high impedance
      addr_received = 0x00;              
      while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready
      //get the address
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << 1;
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << 2;
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << 3;
   }   

   _CL_SetPortBit (chan.ack_p, 0);              // Tell sender address received
   while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 4-phase handshake
   _CL_SetPortBit (chan.ack_p, 1);

   for(i=0;i<8;++i)
   {
      while (_CL_GetPortBit(chan.req_p));          // Wait until data is ready

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data1_p) << i;

      _CL_SetPortBit (chan.ack_p, 0);              // Tell sender data received
      while (!_CL_GetPortBit(chan.req_p));         // Finish the 4-phase handshake
      _CL_SetPortBit (chan.ack_p, 1);
   }

   return char_received;
}

#endif

#if CL_REC_M_D1P4_A

uchar CL_RecMstD1P4A(CL_REC_M_D1_P4_A chan, uchar addr)
{
   ushort data i;
   uchar data char_received=0x00;

   //Put the 4 bit address on the port
   _CL_SetPortBit(chan.addr4_p, (bit)  (addr & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x01, (bit)  (addr >> 1 & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x02, (bit)  (addr >> 2 & 0x01));  
   _CL_SetPortBit(chan.addr4_p+0x03, (bit)  (addr >> 4 & 0x01));  
   
   _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
   while (_CL_GetPortBit(chan.ack_p));          // wait until address received 
   _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 4-phase handshake
   while (!_CL_GetPortBit(chan.ack_p));   

   //Set data port to 1 to read
   _CL_SetPortBit(chan.data1_p,1);

   for(i=0;i<8;++i)
   {
      _CL_SetPortBit (chan.req_p,  0);             // Tell sender ready to receive 
      while (_CL_GetPortBit(chan.ack_p));          // Wait until data is ready

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data1_p) << i;

      _CL_SetPortBit (chan.req_p,  1);             // Tell sender received data
      while (!_CL_GetPortBit(chan.ack_p));         // Finish 4-phase handshake
   }

   return char_received;
}

#endif

#if CL_SEND_S_D1P4_A

void CL_SendSrvD1P4A(CL_SEND_S_D1_P4_A chan, uchar char_to_send)
{
   ushort data i;
   uchar data addr_received=0x00;

   while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready

   //get the address
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p);
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << 1;
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << 2;
   addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << 3;

   while(addr_received != chan.addr)
   {
      _CL_SetPortBit (chan.ack_p, 1);           // set ack to high impedance
      addr_received = 0x00;              
      while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready
      //get the address
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << 1;
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << 2;
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << 3;
   }   

   _CL_SetPortBit (chan.ack_p, 0);              // Tell sender address received
   while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 4-phase handshake
   _CL_SetPortBit (chan.ack_p, 1);

   for(i=0;i<8;++i)
   {
      while (_CL_GetPortBit(chan.req_p));          // Wait for master request data

      //Put the data on the port
      _CL_SetPortBit(chan.data1_p, (bit) ((char_to_send >> i) & 0x01));  

      _CL_SetPortBit (chan.ack_p, 0);              // Tell master data is ready
      while (!_CL_GetPortBit(chan.req_p));         // Wait until master receives 
      _CL_SetPortBit (chan.ack_p, 1);              // Finish the 4-phase handshake
   }

   //Set data port to high impedance
   _CL_SetPortBit(chan.data1_p,1);

   return;
}

#endif

/****************************************************************************** 
    8-bit data bus, 2-phase handshake (D8_P2) (req line only, fixed wait time) 
******************************************************************************/ 

#if CL_SEND_M_D8P2

void CL_SendMstD8P2(CL_SEND_M_D8_P2 chan, uchar char_to_send)
{
   _CL_SetPortByte(chan.data_p, char_to_send);  // Put data on port
   _CL_SetPortBit (chan.req_p,  0);             // Tell receiver data is ready
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
   _CL_SetPortBit (chan.req_p,  1);             // Finish 2-phase handshake
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time   

   _CL_SetPortByte(chan.data_p, 0xff);          // Release the bus
   return;
}

#endif

#if CL_REC_S_D8P2

uchar CL_RecSrvD8P2(CL_REC_S_D8_P2 chan)
{
   uchar data char_received=0x00;

   while (_CL_GetPortBit(chan.req_p));          // Wait until data is ready
   char_received = _CL_GetPortByte(chan.data_p);// Get the data
   while (!_CL_GetPortBit(chan.req_p));         // Finish the 2-phase handshake

   return char_received;
}

#endif

#if CL_REC_M_D8P2

uchar CL_RecMstD8P2(CL_REC_M_D8_P2 chan)
{
   uchar data char_received=0x00;

   _CL_SetPortByte(chan.data_p,0xff);           // Write all 1's before reading 

   _CL_SetPortBit (chan.req_p,  0);             // Tell sender ready to receive 
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
   char_received = _CL_GetPortByte(chan.data_p);// Get the data
   _CL_SetPortBit (chan.req_p,  1);             // Tell sender received data
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time

   return char_received;
}

#endif

#if CL_SEND_S_D8P2

void CL_SendSrvD8P2(CL_SEND_S_D8_P2 chan, uchar char_to_send)
{
   while (_CL_GetPortBit(chan.req_p));          // Wait for master request data
   _CL_SetPortByte(chan.data_p,char_to_send);   // Put the data on the port
   while (!_CL_GetPortBit(chan.req_p));         // Wait until master receives 

   _CL_SetPortByte(chan.data_p,0xff);           // Set data port to high impedance
   return;
}

#endif

#if CL_SEND_M_D8P2_A

void CL_SendMstD8P2A(CL_SEND_M_D8_P2_A chan, uchar addr, uchar char_to_send)
{
   _CL_SetPortByte(chan.addr_p, addr);          // Put address on port
   _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
   _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 2-phase handshake
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time

   _CL_SetPortByte(chan.data_p, char_to_send);  // Put data on port
   _CL_SetPortBit (chan.req_p,  0);             // Tell receiver data is ready
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
   _CL_SetPortBit (chan.req_p,  1);             // Finish 2-phase handshake
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time

   _CL_SetPortByte(chan.data_p, 0xff);          // Release the bus
   return;
}

#endif

#if CL_REC_S_D8P2_A

uchar CL_RecSrvD8P2A(CL_REC_S_D8_P2_A chan)
{
   uchar data char_received=0x00;
   uchar data addr_received=0x00;

   while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready
   addr_received = _CL_GetPortByte(chan.addr_p);// Get the address

   while(addr_received != chan.addr)
   {
      addr_received = 0x00;              
      while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready
      addr_received = _CL_GetPortByte(chan.addr_p);// Get the address
   }   

   while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 2-phase handshake

   while (_CL_GetPortBit(chan.req_p));          // Wait until data is ready
   char_received = _CL_GetPortByte(chan.data_p);// Get the data
   while (!_CL_GetPortBit(chan.req_p));         // Finish the 2-phase handshake

   return char_received;
}

#endif

#if CL_REC_M_D8P2_A

uchar CL_RecMstD8P2A(CL_REC_M_D8_P2_A chan, uchar addr)
{
   uchar data char_received=0x00;

   _CL_SetPortByte(chan.addr_p, addr);          // Put address on port
   _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
   _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 2-phase handshake
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time

   _CL_SetPortByte(chan.data_p,0xff);           // Write all 1's before reading 

   _CL_SetPortBit (chan.req_p,  0);             // Tell sender ready to receive 
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
   char_received = _CL_GetPortByte(chan.data_p);// Get the data
   _CL_SetPortBit (chan.req_p,  1);             // Tell sender received data
   WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time

   return char_received;
}

#endif

#if CL_SEND_S_D8P2_A

void CL_SendSrvD8P2A(CL_SEND_S_D8_P2_A chan, uchar char_to_send)
{
   uchar data addr_received=0x00;

   while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready
   addr_received = _CL_GetPortByte(chan.addr_p);// Get the address

   while(addr_received != chan.addr)
   {
      addr_received = 0x00;              
      while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready
      addr_received = _CL_GetPortByte(chan.addr_p);// Get the address
   }   

   while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 2-phase handshake

   while (_CL_GetPortBit(chan.req_p));          // Wait for master request data
   _CL_SetPortByte(chan.data_p,char_to_send);   // Put the data on the port
   while (!_CL_GetPortBit(chan.req_p));         // Wait until master receives 

   _CL_SetPortByte(chan.data_p,0xff);           // Set data port to high impedance
   return;
}

#endif

/****************************************************************************** 
    4-bit data bus, 2-phase handshake (D4_P2) (req line only, fixed wait time) 
******************************************************************************/ 

#if CL_SEND_M_D4P2

void CL_SendMstD4P2(CL_SEND_M_D4_P2 chan, uchar char_to_send)
{
   ushort data i;

   for(i=0;i<2;++i)
   {
      // Put the data on the port
      _CL_SetPortBit(chan.data4_p, (bit) (char_to_send >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x01, (bit) (char_to_send >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x02, (bit) (char_to_send >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x03, (bit) (char_to_send >> (4*i + 3) & 0x01));  

      _CL_SetPortBit (chan.req_p,  0);          // Tell receiver data is ready
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
      _CL_SetPortBit (chan.req_p,  1);          // Finish 2-phase handshake
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
   }

   //Release the bus
   _CL_SetPortBit(chan.data4_p, 1);
   _CL_SetPortBit(chan.data4_p + 0x01, 1);
   _CL_SetPortBit(chan.data4_p + 0x02, 1);
   _CL_SetPortBit(chan.data4_p + 0x03, 1);

   return;
}

#endif

#if CL_REC_S_D4P2

uchar CL_RecSrvD4P2(CL_REC_S_D4_P2 chan)
{
   ushort data i;
   uchar data char_received=0x00;

   for (i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.req_p));       // Wait until data is ready

      char_received |= (char)_CL_GetPortBit(chan.data4_p) << (4*i);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x01) << (4*i + 1);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x02) << (4*i + 2);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x03) << (4*i + 3);

      while (!_CL_GetPortBit(chan.req_p));      // Finish the 2-phase handshake
   }

   return char_received;
}

#endif

#if CL_REC_M_D4P2

uchar CL_RecMstD4P2(CL_REC_M_D4_P2 chan)
{
   ushort data i;
   uchar data char_received=0x00;

   for(i=0;i<2;++i)
   {
      _CL_SetPortBit (chan.req_p,  0);          // Tell sender ready to receive 
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data4_p) << (4*i);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x01) << (4*i + 1);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x02) << (4*i + 2);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x03) << (4*i + 3);

      _CL_SetPortBit (chan.req_p,  1);          // Tell sender received data
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
   }

   return char_received;
}

#endif

#if CL_SEND_S_D4P2

void CL_SendSrvD4P2(CL_SEND_S_D4_P2 chan, uchar char_to_send)
{
   ushort data i;

   for(i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.req_p));       // Wait for master request data

      //Put the data on the port
      _CL_SetPortBit(chan.data4_p, (bit) (char_to_send >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x01, (bit) (char_to_send >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x02, (bit) (char_to_send >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x03, (bit) (char_to_send >> (4*i + 3) & 0x01));  

      while (!_CL_GetPortBit(chan.req_p));      // Wait until master receives 
   }

   //Set all data port bits to high impedance
   _CL_SetPortBit(chan.data4_p,1);
   _CL_SetPortBit(chan.data4_p+0x01,1); 
   _CL_SetPortBit(chan.data4_p+0x02,1); 
   _CL_SetPortBit(chan.data4_p+0x03,1); 

   return;
}

#endif

#if CL_SEND_M_D4P2_A

void CL_SendMstD4P2A(CL_SEND_M_D4_P2_A chan, uchar addr, uchar char_to_send)
{
   ushort data i;

   //Put the 8 bit address on the port
   for(i=0;i<2;++i)
   {
      _CL_SetPortBit(chan.addr4_p, (bit)  (addr >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.addr4_p+0x01, (bit)  (addr >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.addr4_p+0x02, (bit)  (addr >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.addr4_p+0x03, (bit)  (addr >> (4*i + 3) & 0x01));

      _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
      _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 2-phase handshake
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
   }

   for(i=0;i<2;++i)
   {
      //Put the data on the port
      _CL_SetPortBit(chan.data4_p, (bit) (char_to_send >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x01, (bit) (char_to_send >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x02, (bit) (char_to_send >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x03, (bit) (char_to_send >> (4*i + 3) & 0x01));  
   
      _CL_SetPortBit (chan.req_p,  0);             // Tell receiver data is ready
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
      _CL_SetPortBit (chan.req_p,  1);             // Finish 2-phase handshake
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
   }

   //Release the bus
   _CL_SetPortBit(chan.data4_p, 1);
   _CL_SetPortBit(chan.data4_p + 0x01, 1);
   _CL_SetPortBit(chan.data4_p + 0x02, 1);
   _CL_SetPortBit(chan.data4_p + 0x03, 1);
   
   return;
}

#endif

#if CL_REC_S_D4P2_A

uchar CL_RecSrvD4P2A(CL_REC_S_D4_P2_A chan)
{
   ushort data i;
   uchar data char_received=0x00;
   uchar data addr_received=0x00;

   for(i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready

      //get the address
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p) << (4*i);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << (4*i + 1);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << (4*i + 2);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << (4*i + 3);

      while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 2-phase handshake
   }

   while(addr_received != chan.addr)
   {
      addr_received = 0x00;
      for(i=0;i<2;++i)
      {
         while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready

         //get the address
         addr_received |= (char)_CL_GetPortBit(chan.addr4_p) << (4*i);
         addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << (4*i + 1);
         addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << (4*i + 2);
         addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << (4*i + 3);

         while (!_CL_GetPortBit(chan.addr_req_p));  // Finish the 2-phase handshake
      }
   }   

   for(i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.req_p));          // Wait until data is ready

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data4_p) << (4*i);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x01) << (4*i + 1);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x02) << (4*i + 2);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x03) << (4*i + 3);

      while (!_CL_GetPortBit(chan.req_p));         // Finish the 2-phase handshake
   }

   return char_received;
}

#endif

#if CL_REC_M_D4P2_A

uchar CL_RecMstD4P2A(CL_REC_M_D4_P2_A chan, uchar addr)
{
   ushort data i;
   uchar data char_received=0x00;

   //Put the 8 bit address on the port
   for(i=0;i<2;++i)
   {
      _CL_SetPortBit(chan.addr4_p, (bit)  (addr >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.addr4_p+0x01, (bit)  (addr >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.addr4_p+0x02, (bit)  (addr >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.addr4_p+0x03, (bit)  (addr >> (4*i + 3) & 0x01));

      _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
      _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 2-phase handshake
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
   }
   
   //Set all data port bits to 1 to read
   _CL_SetPortBit(chan.data4_p,1);
   _CL_SetPortBit(chan.data4_p+0x01,1); 
   _CL_SetPortBit(chan.data4_p+0x02,1); 
   _CL_SetPortBit(chan.data4_p+0x03,1); 

   for(i=0;i<2;++i)
   {
      _CL_SetPortBit (chan.req_p,  0);             // Tell sender ready to receive 
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data4_p) << (4*i);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x01) << (4*i + 1);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x02) << (4*i + 2);
      char_received |= (char)_CL_GetPortBit(chan.data4_p+0x03) << (4*i + 3);

      _CL_SetPortBit (chan.req_p,  1);             // Tell sender received data
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
   }

   return char_received;
}

#endif

#if CL_SEND_S_D4P2_A

void CL_SendSrvD4P2A(CL_SEND_S_D4_P2_A chan, uchar char_to_send)
{
   ushort data i;
   uchar data addr_received=0x00;

   for(i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready

      //get the address
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p) << (4*i);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << (4*i + 1);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << (4*i + 2);
      addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << (4*i + 3);

      while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 2-phase handshake
   }

   while(addr_received != chan.addr)
   {
      addr_received = 0x00;
      for(i=0;i<2;++i)
      {
         while (_CL_GetPortBit(chan.addr_req_p));  // Wait until address is ready

         //get the address
         addr_received |= (char)_CL_GetPortBit(chan.addr4_p) << (4*i);
         addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x01) << (4*i + 1);
         addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x02) << (4*i + 2);
         addr_received |= (char)_CL_GetPortBit(chan.addr4_p+0x03) << (4*i + 3);

         while (!_CL_GetPortBit(chan.addr_req_p));  // Finish the 2-phase handshake
      }
   }   

   for(i=0;i<2;++i)
   {
      while (_CL_GetPortBit(chan.req_p));          // Wait for master request data

      //Put the data on the port
      _CL_SetPortBit(chan.data4_p, (bit) (char_to_send >> (4*i) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x01, (bit) (char_to_send >> (4*i + 1) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x02, (bit) (char_to_send >> (4*i + 2) & 0x01));  
      _CL_SetPortBit(chan.data4_p+0x03, (bit) (char_to_send >> (4*i + 3) & 0x01));  

      while (!_CL_GetPortBit(chan.req_p));         // Wait until master receives 
   }

   //Set all data port bits to high impedance
   _CL_SetPortBit(chan.data4_p,1);
   _CL_SetPortBit(chan.data4_p+0x01,1); 
   _CL_SetPortBit(chan.data4_p+0x02,1); 
   _CL_SetPortBit(chan.data4_p+0x03,1); 

   return;
}

#endif


/****************************************************************************** 
    1-bit data bus, 2-phase handshake (D1_P2) (req line only, fixed wait time) 
******************************************************************************/ 

#if CL_SEND_M_D1P2

void CL_SendMstD1P2(CL_SEND_M_D1_P2 chan, uchar char_to_send)
{
   ushort data i;

   for(i=0;i<8;++i)
   {
      // Put the data on the port
      _CL_SetPortBit(chan.data1_p, (bit) ((char_to_send >> i) & 0x01));  

      _CL_SetPortBit (chan.req_p,  0);          // Tell receiver data is ready
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
      _CL_SetPortBit (chan.req_p,  1);          // Finish 2-phase handshake
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
   }

   //Set data port to high impedance
   _CL_SetPortBit(chan.data1_p,1);

   return;
}

#endif

#if CL_REC_S_D1P2

uchar CL_RecSrvD1P2(CL_REC_S_D1_P2 chan)
{
   ushort data i;
   uchar data char_received=0x00;

   for (i=0;i<8;++i)
   {
      while (_CL_GetPortBit(chan.req_p));       // Wait until data is ready

      char_received |= (char)_CL_GetPortBit(chan.data1_p) << i;

      while (!_CL_GetPortBit(chan.req_p));      // Finish the 2-phase handshake
   }

   return char_received;
}

#endif

#if CL_REC_M_D1P2

uchar CL_RecMstD1P2(CL_REC_M_D1_P2 chan)
{
   ushort data i;
   uchar data char_received=0x00;

   for(i=0;i<8;++i)
   {
      _CL_SetPortBit (chan.req_p,  0);          // Tell sender ready to receive 
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data1_p) << i;

      _CL_SetPortBit (chan.req_p,  1);          // Tell sender received data
      WaitFor(chan.wait_time,chan.oneus_loop);  // Fixed wait time
   }

   return char_received;
}

#endif

#if CL_SEND_S_D1P2

void CL_SendSrvD1P2(CL_SEND_S_D1_P2 chan, uchar char_to_send)
{
   ushort data i;

   for(i=0;i<8;++i)
   {
      while (_CL_GetPortBit(chan.req_p));       // Wait for master request data

      //Put the data on the port
      _CL_SetPortBit(chan.data1_p, (bit) ((char_to_send >> i) & 0x01));  

      while (!_CL_GetPortBit(chan.req_p));      // Wait until master receives
   }

   //Set the data port to high impedance
   _CL_SetPortBit(chan.data1_p,1);

   return;
}

#endif

#if CL_SEND_M_D1P2_A

void CL_SendMstD1P2A(CL_SEND_M_D1_P2_A chan, uchar addr, uchar char_to_send)
{
   ushort data i;

   //Put the 8 bit address on the port
   for(i=0;i<8;++i)
   {
      _CL_SetPortBit(chan.addr1_p, (bit) ((addr >> i) & 0x01));
      _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
      WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
      _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 2-phase handshake
      WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
   }

   for(i=0;i<8;++i)
   {
      //Put the data on the port
      _CL_SetPortBit(chan.data1_p, (bit) ((char_to_send >> i) & 0x01));  
      WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time   
      _CL_SetPortBit (chan.req_p,  0);             // Tell receiver data is ready
      WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
      _CL_SetPortBit (chan.req_p,  1);             // Finish 2-phase handshake

   }

   //Set data port to high impedance
   _CL_SetPortBit(chan.data1_p,1);

   return;
}

#endif

#if CL_REC_S_D1P2_A

uchar CL_RecSrvD1P2A(CL_REC_S_D1_P2_A chan)
{
   ushort data i;
   uchar data char_received=0x00;
   uchar data addr_received=0x00;

   for(i=0;i<8;++i)
   {
       while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready
       // Get the address
       addr_received |= (char)_CL_GetPortBit(chan.addr1_p) << i;
       while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 2-phase handshake
   }

   while(addr_received != chan.addr)
   {
      addr_received = 0x00;
      for(i=0;i<8;++i)
      {
          while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready
          // Get the address
          addr_received |= (char)_CL_GetPortBit(chan.addr1_p) << i;
          while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 2-phase handshake
       }
   }

   for(i=0;i<8;++i)
   {
      while (_CL_GetPortBit(chan.req_p));          // Wait until data is ready

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data1_p) << i;

      while (!_CL_GetPortBit(chan.req_p));         // Finish the 2-phase handshake
   }

   return char_received;
}

#endif

#if CL_REC_M_D1P2_A

uchar CL_RecMstD1P2A(CL_REC_M_D1_P2_A chan, uchar addr)
{
   ushort data i;
   uchar data char_received=0x00;

   //Put the 8 bit address on the port
   for(i=0;i<8;++i)
   {
      _CL_SetPortBit(chan.addr1_p, (bit) ((addr >> i) & 0x01));
      _CL_SetPortBit (chan.addr_req_p, 0);         // Tell receivers to check address
      WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
      _CL_SetPortBit (chan.addr_req_p, 1);         // Finish 2-phase handshake
      WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
   }

   //Set data port to 1 to read
   _CL_SetPortBit(chan.data1_p,1);

   for(i=0;i<8;++i)
   {
      _CL_SetPortBit (chan.req_p,  0);             // Tell sender ready to receive 
      WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time

      //Get the data
      char_received |= (char)_CL_GetPortBit(chan.data1_p) << i;

      _CL_SetPortBit (chan.req_p,  1);             // Tell sender received data
      WaitFor(chan.wait_time,chan.oneus_loop);     // Fixed wait time
   }

   return char_received;
}

#endif

#if CL_SEND_S_D1P2_A

void CL_SendSrvD1P2A(CL_SEND_S_D1_P2_A chan, uchar char_to_send)
{
   ushort data i;
   uchar data addr_received=0x00;

   for(i=0;i<8;++i)
   {
       while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready
       // Get the address
       addr_received |= (char)_CL_GetPortBit(chan.addr1_p) << i;
       while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 2-phase handshake
   }

   while(addr_received != chan.addr)
   {
      addr_received = 0x00;
      for(i=0;i<8;++i)
      {
          while (_CL_GetPortBit(chan.addr_req_p));     // Wait until address is ready
          // Get the address
          addr_received |= (char)_CL_GetPortBit(chan.addr1_p) << i;
          while (!_CL_GetPortBit(chan.addr_req_p));    // Finish the 2-phase handshake
       }
   }

   for(i=0;i<8;++i)
   {
      while (_CL_GetPortBit(chan.req_p));          // Wait for master request data

      //Put the data on the port
      _CL_SetPortBit(chan.data1_p, (bit) ((char_to_send >> i) & 0x01));  

      while (!_CL_GetPortBit(chan.req_p));         // Wait until master receives 
   }

   //Set data port to high impedance
   _CL_SetPortBit(chan.data1_p,1);

   return;
}

#endif

/******************************************************************************

   PC-Serial (PCS) communications 

******************************************************************************/

#if CL_S_PCSERIAL

static void baudset(uint baudrate, float freq)
{
   TH1=(int)(256-((freq*1000000/384)/baudrate));
   TL1=TH1;
   TMOD=0x20;  // timer 1 in mode 2, auto-reload
   TR1=1;
   SCON=0x40;  // 8 bit UART, receive off 
}

#endif

#if CL_SEND_S_PCSERIAL

static void _CL_InitSendPCSMc(void)
{
   IE=0;       // disable all interrupts, using polling

   SMOD=0;     // for calculation of baud rate
   TR0=0;      // both the timers are off, initially
   TR1=0;
}        

#endif

#if (CL_SEND_S_PCSERIAL==1)

void CL_InitDefaultSendSrvPCS(CL_SEND_S_PCS *chan, float freq)
{
   chan.dsr_p	= _CL_default_dsr_p;
   chan.dtr_p	= _CL_default_dtr_p;
   chan.cts_p	= _CL_default_cts_p;
   chan.rts_p	= _CL_default_rts_p;

   _CL_InitSendPCSMc();
   baudset(2400,freq);          // set the default baud rate

   _CL_SetPortBit(chan.dtr_p,1);
   _CL_SetPortBit(chan.rts_p,1); // initialize the incoming handshake signals,inactive
}

#endif

#if (CL_SEND_S_PCSERIAL==2)

void CL_InitSendSrvPCS(CL_SEND_S_PCS* chan, dsr_p_address, dtr_p_address,
   cts_p_address, rts_p_address, uint baudrate, float freq) 
{
   chan.dsr_p   = dsr_p_address;
   chan.dtr_p   = dtr_p_address;
   chan.cts_p   = cts_p_address;
   chan.rts_p   = rts_p_address;

   _CL_InitSendPCSMc();
   baudset(baudrate, freq);      // set the specified baud rate
   _CL_SetPortBit(chan.dtr_p,1); 
   _CL_SetPortBit(chan.rts_p,1); // initialize the incoming handshake signals,inactive
}

#endif

#if CL_SEND_S_PCSERIAL

void CL_SendSrvPCS(CL_SEND_S_PCS chan, uchar message)
{
   // Wait for the primary handshake, active low
   while(_CL_GetPortBit(chan.dtr_p));
   
   // Wait for the character to be transmitted.
   // Although TI can be used for the purpose, it can take a while to go low
   // With the PC, this can make the 8051 miss the RTS incoming pulse
   // So it is better to ignore TI and wait rts to go low 
   
   SBUF=message;
   //while(!TI);

   // Wait for the secondary handshake, active low
   while(_CL_GetPortBit(chan.rts_p));

   //Set TI back to 0
   TI=0;
}

#endif

#if CL_REC_S_PCSERIAL

static void _CL_InitRecPCSMc(void)
{
   IE=0;       // disable all interrupts, using polling

   SMOD=0;     // for calculation of baud rate
   TR0=0;      // both the timers are off, initially
   TR1=0;
}        

#endif

#if (CL_REC_S_PCSERIAL==1)

void CL_InitDefaultRecSrvPCS(CL_REC_S_PCS* chan, float freq)
{
   chan.dsr_p	= _CL_default_dsr_p;
   chan.dtr_p	= _CL_default_dtr_p;
   chan.cts_p	= _CL_default_cts_p;
   chan.rts_p	= _CL_default_rts_p;

   _CL_InitRecPCSMc();
   baudset(2400, freq);      // set the default baud rate
   
   _CL_SetPortBit(chan.dsr_p,1); 
   _CL_SetPortBit(chan.cts_p,1); // initialize the outgoing handshake signals, inactive

}

#endif

#if (CL_REC_S_PCSERIAL==2)

void CL_InitRecSrvPCS(CL_REC_S_PCS* chan, dsr_p_address, dtr_p_address,
   cts_p_address, rts_p_address, uint baudrate, float freq) 
{
   chan.dsr_p   = dsr_p_address;
   chan.dtr_p   = dtr_p_address;
   chan.cts_p   = cts_p_address;
   chan.rts_p   = rts_p_address;

   _CL_InitRecPCSMc();
   baudset(baudrate, freq);      // set the specified baud rate

   _CL_SetPortBit(chan.dsr_p,1);
   _CL_SetPortBit(chan.cts_p,1); // initialize the outgoing handshake signals, inactive
  
}

#endif

#if CL_REC_S_PCSERIAL

uchar CL_RecSrvPCS(CL_REC_S_PCS chan)
{
   uchar        char_received;
   // ushort       wait = 50;
 
   // enable reception, if not already done so  
   REN=1;

   // Set the primary handshake
   _CL_SetPortBit(chan.dsr_p,0); 
 
   //Wait for character to be received and then read the character 
   while(!RI);
   char_received=SBUF;
 
   // Then set the secondary handshake
   _CL_SetPortBit(chan.cts_p,0); 

   // Reset RI
   RI=0;

   // Next turn off handshake signals
   _CL_SetPortBit(chan.dsr_p,1);
   _CL_SetPortBit(chan.cts_p,1);

   // Turn off reception
   REN=0;

   return char_received;
}

#endif
