library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;

entity iec_common_block is
  port ( RESET           : in std_logic;
         CLK             : in std_logic;
         IECIN           : in std_logic;
         B_PREAMBLE      : in std_logic;
         M_PREAMBLE      : in std_logic;
         W_PREAMBLE      : in std_logic;
         CNT_CLR         : in std_logic;
         DATA_BIT_READY  : in std_logic;
         SHIFT_VAL_OUT   : out std_logic_vector(0 to 32);
         CNT_OUT         : out std_logic_vector(3 downto 0);
         B_FLAG          : out std_logic;
         LEFT_RIGHT      : out std_logic;
         DATA_LOAD       : out std_logic);
end iec_common_block;

architecture beh of iec_common_block is

  type stat_type  is (INIT, B_STAT, M_STAT, W_STAT, WAIT_STAT);
  signal curr_stat, next_stat  : stat_type;
  signal shift_val  : std_logic_vector(0 to 32);
  signal data_load_tmp  : std_logic;
  signal cnt  : std_logic_vector(3 downto 0);
  signal cnt_ready, cnt_reset  : std_logic;
  signal data_cnt  : std_logic_vector(4 downto 0);
  signal data_bit_end  : std_logic;

begin

  shift_reg: process (RESET, CLK)
  begin
    if (RESET = '0') then
      shift_val <= (others => '0');
    elsif (CLK'event and CLK = '1') then
      shift_val(32) <= IECIN;
      shift_val(0 to 31) <= shift_val(1 to 32);
    end if;
  end process shift_reg;

  cnt_logic: process (RESET, CLK)
  begin
    if (RESET = '0') then
      cnt <= (others => '0');
    elsif (CLK'event and CLK = '1') then
      if (cnt_reset = '1') then
        cnt <= (others => '0');
      elsif (cnt_ready = '1') then
        if (CNT_CLR = '1') then
          cnt <= (others => '0');
        else
          cnt <= cnt + '1';
        end if;
      end if;
    end if;
  end process cnt_logic;

  buf: process (RESET, CLK)
  begin
    if (RESET = '0') then
      data_load_tmp <= '0';
    elsif (CLK'event and CLK = '1') then
      if (data_load_tmp = '1') then
        data_load_tmp <= '0';
      else
        data_load_tmp <= data_bit_end;
      end if;
    end if;
  end process buf;

  left_right_pro: process (RESET, CLK)
  begin
    if (RESET = '0') then
      B_FLAG <= '0';
      LEFT_RIGHT <= '1';
    elsif (CLK'event and CLK = '1') then
      if (curr_stat = B_STAT or curr_stat = M_STAT) then
        if (curr_stat = B_STAT) then
          B_FLAG <= '1';
        else
          B_FLAG <= '0';
        end if;
        LEFT_RIGHT <= '1';
      elsif (curr_stat = W_STAT) then
        B_FLAG <= '0';
        LEFT_RIGHT <= '0';
      end if;
    end if;
  end process left_right_pro;

-- -------------------------------------------------------------
  sync: process (RESET, CLK)
  begin
    if (RESET = '0') then
      curr_stat <= INIT;
    elsif (CLK'event and CLK = '1') then
      curr_stat <= next_stat;
    end if;
  end process sync;

  comb: process (curr_stat, B_PREAMBLE, M_PREAMBLE, W_PREAMBLE,
                 data_bit_end)
  begin
    case curr_stat is
      when INIT =>
        if (B_PREAMBLE = '1') then
          next_stat <= B_STAT;
        else
          next_stat <= INIT;
        end if;
      when B_STAT =>
        if (data_bit_end = '1') then
          next_stat <= WAIT_STAT;
        else
          next_stat <= B_STAT;
        end if;
      when M_STAT =>
        if (data_bit_end = '1') then
          next_stat <= WAIT_STAT;
        else
          next_stat <= M_STAT;
        end if;
      when W_STAT =>
        if (data_bit_end = '1') then
          next_stat <= WAIT_STAT;
        else
          next_stat <= W_STAT;
        end if;
      when WAIT_STAT =>
        if (B_PREAMBLE = '1') then
          next_stat <= B_STAT;
        elsif (M_PREAMBLE = '1') then
          next_stat <= M_STAT;
        elsif (W_PREAMBLE = '1') then
          next_stat <= W_STAT;
        else
          next_stat <= WAIT_STAT;
        end if;
    end case;
  end process comb;

  data_cnt_pro: process (RESET, CLK)
  begin
    if (RESET = '0') then
      data_cnt <= (others => '0');
    elsif (CLK'event and CLK = '1') then
      if (cnt_reset = '1') then
        data_cnt <= (others => '0');
      elsif (cnt_ready = '1' and DATA_BIT_READY = '1') then
        data_cnt <= data_cnt + '1';
      end if;
    end if;
  end process data_cnt_pro;
-- -------------------------------------------------------------

  data_bit_end <= '1' when (data_cnt = X"1C") else '0';  -- 0x1C = 28
  cnt_ready <= '1' when (curr_stat = B_STAT or curr_stat = M_STAT or
                         curr_stat = W_STAT) else '0';
  cnt_reset <= '1' when (curr_stat = INIT or curr_stat = WAIT_STAT) else '0';

  SHIFT_VAL_OUT <= shift_val;
  CNT_OUT <= cnt;
  DATA_LOAD <= data_load_tmp;

end beh;
